import base64
import shutil
from pathlib import Path

import safer

from ..classes.errors.db import FileAlreadyExistsError, WriteInDatabaseError
from ..classes import Document
from . import qaapDB

class FlatFileDB(qaapDB):
    """
    A class to manage data in text files using a structured folder hierarchy.

    This class is inspired by the structure of the Grav CMS pages. It creates a structured folder hierarchy
    to store documents, notes, and other data. The root directory is specified during initialization.
    The class provides methods to check, get, and write documents, notes, and other data to the database.

    The folder and file structure must be as follows:
    <db_name>/
    ├── documents/
    │   ├── <document_title>/
    │   │   ├── document.txt
    │   │   ├── media1.png *
    │   │   ├── media2.png *
    │   │   └── <note_title>/ *
    │   │       ├── note.txt
    │   │       └── media.png *
    │   ├── ...
    ├── summaries/
    │   ├── # catalog.json
    │   └── # <attribute_name>.txt
    └── rag/
        └── # store.pkl

    Where every value between <> are variables, * indicates optional values 
    and # indicates files that are generated by the app.

    Attributes:
        root (str): The root directory path where the folder hierarchy will be created.
    """

    def __init__(self, root: int):
        """
            Initializes a FlatFileDB instance (inherits from qAApDB), which manages data in text files using a structured folder hierarchy.

            Args:
                root (str): The root directory path where the folder hierarchy will be created.
                        All data files and subdirectories will be stored under this path.
        """
        # ottoDB.__init__(self)
        self.root = root
        self._setup_db_folder()

    def _setup_db_folder(self):
        """
            Initializes the required database folder structure and files.

            Creates the following directories under `self.root` if they do not exist:
            - "rag"
            - "documents"
            - "summaries"
        """

        # Create directories
        Path(self.root, "rag").mkdir(parents=True, exist_ok=True)
        Path(self.root, "documents").mkdir(parents=True, exist_ok=True)
        Path(self.root, "summaries").mkdir(parents=True, exist_ok=True)

    
    # ====================================================================== CHECK METHODS

    def document_exists(self, document_title: str) -> bool:
        """
            Checks if a directory for the specified document exists.

            Args:
                document_title (str): Name of the document to check.

            Returns:
                bool: True if the document exists, False otherwise.
        """
        path = Path(self.root,"documents",document_title)

        return path.exists()

    
    def note_exists(self, document_title: str, note_title: str) -> bool:
        """ 
            Checks if a note for the specified document by the specified note_title exists.
            
            Args:
                document_title (str): Name of the document.
                note_title (str): Name of the note_title.

            Returns:
                bool: True if the note exists, False otherwise.
        """
        path = Path(self.root,"documents",document_title,note_title)
        return path.exists()


    def attribute_exists(self, attribute: str) -> bool:
        """ 
            Checks if a attribute exists.

            Args:
                attribute (str): Name of the attribute to check.

            Returns:
                bool: True if the attribute exists, False otherwise.

            Raises:
                FileNotFoundError: If the <attribute>.txt file does not exist.
        """
        result = False

        with open(Path(self.root,"summaries",f"{attribute}.txt"),encoding="utf-8") as file:
            content = file.read()
            lines = content.splitlines()
            result = attribute in lines

        return result

    # ======================================================================== GET METHODS

    def get_catalog(self) -> str:
        """
            Retrieves the catalog.

            Returns:
                str: JSON string representing the catalog.

            Raises:
                FileNotFoundError: If the catalog.json file does not exist.
        """
        try:
            with open(Path(self.root,"summaries","catalog.json"),encoding="utf-8") as file:
                content = file.read()
                return content
        except Exception as e:
            raise FileNotFoundError(str(e))
    

    def get_attribute_values(self, attribute: str) -> str:
        """
            Retrieves the specified attribute values.

            Returns:
                str: The attribute name

            Raises:
                FileNotFoundError: If the <attribute>.txt file does not exist.
        """

        try:
            with open(Path(self.root,"summaries",f"{attribute}.txt"),encoding="utf-8") as file:
                content = file.read()
                return content
        except Exception as e:
            raise FileNotFoundError(str(e))

    
    def get_document(self, document_title: str) -> str:
        """
            Retrieves the content and icon of a specified document.

            Args:
                document_title (str): Name of the document to retrieve.

            Returns:
                tuple[str,str]: A tuple containing the document content (and the icon in base64 format if it exists or an empty string).
            
            Raises:
                FileNotFoundError: If the document does not exist or if the document.txt is missing.
        """
        if not self.document_exists(document_title):
            raise FileNotFoundError()
        
        document_content = ""
        
        try:
            with open(Path(self.root,"documents",document_title,"document.txt"),encoding="utf-8") as file:
                document_content = file.read()

        except Exception as e:
            raise FileNotFoundError(str(e))

        return document_content

    
    def get_document_medias(self, document_title: str, includes: list[str] = None) -> list[str]:
        """
            Retrieves the base64-encoded images for a specified document.
            Returns an empty list if there are no images.

            Args:
                document_title (str): Name of the document.
                includes (list[str]): List of image file stems to include (e.g., ["screen1", "screen2"]).
            
            Returns:
                list[str]: A list of base64-encoded images for the document.

            Raises:
                FileNotFoundError: If the document does not exist.
        """
        if not self.document_exists(document_title):
            raise FileNotFoundError()
        
        paths = sorted(Path(self.root,"documents",document_title).glob("*.png"))

        if includes:
            paths = [path for path in list(paths) if path.stem in includes]

        files = []

        for path in paths:
            with open(path,"rb") as file:
                files.append(base64.urlsafe_b64encode(file.read()).decode("utf-8"))

        return files

    
    def get_notes_for_document(self, document_title: str, perpage: int = 0, page: int = 0) -> list[tuple[str,str,str]]:
        """
            Retrieves all notes for a specified document.
            If perpage is greater than 0, it returns only the notes for the specified page (0-indexed). 
            If perpage less than 1, it returns all notes.
            Returns an empty list if there are no notes.

            Args:
                document_title (str): Name of the document to retrieve notes for.
                perpage (int): Number of notes per page. Default is 0 (all notes).
                page (int): Page number to retrieve (0-indexed). Default is 0.

            Returns:    
                list[tuple[str,str,str]]: A list of tuples, each containing:
                    - The note content as a string.  
                    - The note_title name as a string.
                    - The document name as a string.    

            Raises:
                FileNotFoundError: If the document does not exist or if there are no notes.
        """
        if not self.document_exists(document_title):
            raise FileNotFoundError()
        
        notes = sorted(Path(self.root,"documents",document_title,"").rglob("note.txt"))
        
        if len(notes) == 0:
            return []

        if perpage > 0:
            start = page * perpage
            end = min(len(notes), (page + 1) * perpage)
            if start > len(notes):
                return []
            files = notes[start:end]
        else:
            files = notes

        try:
            result = []
            for file in files:
                with open(file, "r", encoding="utf-8") as f:
                    content = f.read()
                    result.append((content,file.parent.name,file.parent.parent.name))
            return result
        except Exception as e:
            raise FileNotFoundError(str(e))

    def get_note_medias(self, document_title: str, note_title: str, includes: list[str] = None) -> list[str]:
        """
            Retrieves the base64-encoded images for a specified note of a document.
            Returns an empty list if there are no images.

            Args:
                document_title (str): Name of the document.
                note_title (str): Name of the note_title.
                includes (list[str]): List of image file stems to include (e.g., ["screen1", "screen2"]).

            Returns:
                list[str]: A list of base64-encoded images for the note.

            Raises:
                FileNotFoundError: If the note does not exist.
        """
        if not self.note_exists(document_title, note_title):
            raise FileNotFoundError()

        paths = sorted(Path(self.root, "documents", document_title, note_title).glob("*.png"))

        if includes:
            paths = [path for path in list(paths) if path.stem in includes]

        files = []

        for path in paths:
            with open(path, "rb") as file:
                files.append(base64.urlsafe_b64encode(file.read()).decode("utf-8"))

        return files

    # ====================================================================== WRITE METHODS

    def write_catalog(self, json: str) -> bool:
        """
            Writes the catalog to a JSON file.

            Args:
                json (str): JSON string representing the catalog.
            
            Returns:
                bool: True if the write operation was successful
            
            Raises:
                WriteInDatabaseError: If there is an error writing to the file.
        """
        try:
            with safer.open(Path(self.root,"summaries","catalog.json"), "w", encoding="utf-8") as file:
                file.write(json)
            return True
        except Exception as e:
            raise WriteInDatabaseError(str(e))
        
    
    def write_document(self, document_title: str, content: str, medias: list[(str,str)] = []) -> bool:
        """
            Writes a new document to the database.
            It creates a directory for the document, writes the document content to a text file,
            saves the icon as a PNG file, and saves screenshots as PNG files.
            If the document already exists, it raises a FileAlreadyExistsError. 

            Args:
                document_title (str): Name of the document to write.
                content (str): Content of the document.
                icon (str): Base64 encoded icon of the document.
                screenshots (list[str]): list of base64 encoded screenshots of the document.
            
            Returns:
                bool: True if the write operation was successful
            
            Raises:
                FileAlreadyExistsError: If the document already exists.
                WriteInDatabaseError: If there is an error writing to the file or creating directories.
        """
        if self.document_exists(document_title):
            raise FileAlreadyExistsError()
        
        folder_path = Path(self.root,"documents",document_title)

        try:
            folder_path.mkdir(parents=True, exist_ok=False)
            
            with safer.open(Path(folder_path,"document.txt"), "w", encoding="utf-8") as file:
                file.write(content)

            for media in medias:
                with safer.open(Path(folder_path,f"{media[0]}.png"), "wb") as file:
                    file.write(base64.b64decode(media[1]))

            return True
        
        except Exception as e:
            try:
                shutil.rmtree(folder_path)
            except Exception as e:
                raise WriteInDatabaseError(str(e))
            raise WriteInDatabaseError(str(e))

    
    def write_note(self, document_title: str, note_title: str, content: str, medias: list[(str,str)] = []) -> bool:
        """
            Writes a note for a specified document by a specified note_title.
            It creates a directory for the note, writes the note content to a text file,
            saves the note icon as a PNG file, and saves screenshots as PNG files.
            If the document does not exist, it raises a FileNotFoundError.
            If the note already exists, it raises a FileAlreadyExistsError.

            Args:
                document_title (str): Name of the document to write the note for.
                note_title (str): Name of the note_title.
                content (str): Content of the note.
                icon (str): Base64 encoded icon of the note.
                screenshots (list[str]): list of base64 encoded screenshots of the note.
            
            Returns:
                bool: True if the write operation was successful
            
            Raises:
                FileNotFoundError: If the document does not exist.
                FileAlreadyExistsError: If the note already exists.
                WriteInDatabaseError: If there is an error writing to the file or creating directories.
        """
        if not self.document_exists(document_title):
            raise FileNotFoundError()
        
        if self.note_exists(document_title, note_title):
            raise FileAlreadyExistsError()
        
        folder_path = Path(self.root,"documents",document_title,note_title)

        try:
            folder_path.mkdir(parents=True, exist_ok=False)

            with safer.open(Path(folder_path,"note.txt"), "w", encoding="utf-8") as file:
                file.write(content)

            for media in medias:
                with safer.open(Path(folder_path,f"{media[0]}.png"), "wb") as file:
                    file.write(base64.b64decode(media[1]))

            return True
        
        except Exception as e:
            try:
                shutil.rmtree(folder_path)
            except Exception as e:
                raise WriteInDatabaseError(str(e))
            raise WriteInDatabaseError(str(e))

    
    def write_attribute(self, attribute: str, data: str) -> bool:
        """
            Writes new content in the <attributes>.txt file.
            If there is an error writing to the file, it raises a WriteInDatabaseError.

            Args:
                attribute (str):    The attribute name
                data (str):         The list of attributes to write. One attribute per line.
            
            Returns:
                bool: True if the write operation was successful
            
            Raises:
                WriteInDatabaseError: If there is an error writing to the file.
        """
        try:
            with safer.open(Path(self.root,"summaries",f"{attribute}.txt"), "w", encoding="utf-8") as file:
                file.write(f"{data}\n")
            return True
        except Exception as e:
            raise WriteInDatabaseError(str(e))
        
    def add_attribute_values(self, attribute: str, data: list[str]) -> bool:
        """
            Writes a new attribute to the <attribute>.txt file.
            If there is an error writing to the file, it raises a WriteInDatabaseError.

            Args:
                attribute (str):    Name of the attribute to write.
                data (str):         The new attribute value to write.
            
            Returns:
                bool: True if the write operation was successful
            
            Raises:
                FileAlreadyExistsError: If the attribute already exists.
                WriteInDatabaseError: If there is an error writing to the file.
        """
        
        try:
            values = self.get_attribute_values(attribute).splitlines()
            values.append(data)
            unique_values = list(set(values))
            return self.write_attribute(attribute, "\n".join(unique_values))
        except Exception as e:
            raise WriteInDatabaseError(str(e))

    # ======================================================================== RAG METHODS

    def get_all_documents_data(self) -> list[dict[str,str]]:
        """
            Retrieves data from all public Document files.

            This method scans the 'documents' directory for files named 'document.txt', reads their content,
            and compiles the data into a list of dictionaries. Each dictionary contains the content
            of a Document file and the Document name.

            Returns:
                list[dict[str,str]]: A list of dictionaries where each dictionary contains:
                    - "page_content": The content of the document file.
                    - "title": The name of the directory containing the document file. i.e. the document title
        """
        
        docs = []

        documents_paths = sorted(Path(self.root,"documents").rglob("document.txt"))
        for path in documents_paths:
            with open(path,encoding="utf-8") as document:
                title = path.parent.name
                content = document.read()
                document_object: Document = Document.from_text(title,content)
                docs.append(
                    {
                        "content":content,
                        "title": title,
                        "metadata": document_object.metadatas
                    }
                )
                
        return docs

    def get_vector_store(self) -> bytes:
        """
            Retrieves the vector store as binary from a file.

            Returns:
                bytes: The bytes representation of the vectorstore.

            Raises:
                FileNotFoundError: If the file is not found or any other exception occurs during the file read operation.
        """
        try:
            with open(Path(self.root,"rag","store.pkl"), "rb") as file:
                bytes = file.read()
            return bytes
        except Exception as e:
            raise FileNotFoundError(str(e))

    def write_vector_store(self,bytes: bytes) -> bool:
        """
            Writes the binary representation of a vectorstore in a file.

            Args:
                bytes: The bytes representation of the vectorstore.

            Returns:
                bool: True if the write operation is successful.

            Raises:
                WriteInDatabaseError: If any exception occurs during the file write operation.
        """
        try:
            with safer.open(Path(self.root,"rag","store.pkl"), "wb") as file:
                file.write(bytes)
            return True
        except Exception as e:
            raise WriteInDatabaseError(str(e))