"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Generate FEATURE_FLAGS.md, a report of all current feature flags
 */
const fs_1 = require("fs");
const path = require("path");
const feats = require("../lib/features");
const flag_modeling_1 = require("../lib/private/flag-modeling");
async function main() {
    await updateMarkdownFile(path.join(__dirname, '..', 'FEATURE_FLAGS.md'), {
        table: flagsTable(),
        details: flagsDetails(),
        json: recommendedJson(),
        removed: removedFlags(),
        diff: changedFlags(),
        migratejson: migrateJson(),
    });
}
function flagsTable() {
    return renderTable([
        ['Flag', 'Summary', 'Since', 'Type'],
        ...v2flags().map(([name, flag]) => [
            renderLink(mdEsc(name), githubHeadingLink(flagDetailsHeading(name, flag))),
            flag.summary,
            flag.introducedIn.v2 ?? '',
            renderType(flag.type),
        ]),
    ]);
}
function removedFlags() {
    const removedInV2 = flags(flag => flag.introducedIn.v2 === undefined && flag.introducedIn.v1 !== undefined);
    return renderTable([
        ['Flag', 'Summary', 'Type', 'Since'],
        ...removedInV2.map(([name, flag]) => [
            renderLink(mdEsc(name), githubHeadingLink(flagDetailsHeading(name, flag))),
            flag.summary,
            renderType(flag.type),
            flag.introducedIn.v1 ?? '',
        ]),
    ]);
}
function changedFlags() {
    const changedInV2 = flags(flag => !!flag.defaults?.v2 && !!flag.introducedIn.v2);
    return renderTable([
        ['Flag', 'Summary', 'Type', 'Since', 'v1 default', 'v2 default'],
        ...changedInV2.map(([name, flag]) => [
            renderLink(mdEsc(name), githubHeadingLink(flagDetailsHeading(name, flag))),
            flag.summary,
            renderType(flag.type),
            flag.introducedIn.v1 ?? '',
            renderValue(false),
            renderValue(flag.defaults?.v2),
        ]),
    ]);
}
function migrateJson() {
    const changedInV2 = flags(flag => !!flag.defaults?.v2 && !!flag.introducedIn.v2);
    const context = Object.fromEntries(changedInV2.map(([name, _]) => [name, false]));
    return [
        '```json',
        JSON.stringify({ context }, undefined, 2),
        '```',
    ].join('\n');
}
function flagsDetails() {
    const allFlags = flags(_ => true);
    return allFlags.flatMap(([name, flag]) => [
        `### ${flagDetailsHeading(name, flag)}`,
        '',
        `*${flag.summary}* ${renderType(flag.type)}`,
        '',
        dedent(flag.detailsMd),
        '',
        renderTable([
            ['Since', 'Default', 'Recommended'],
            // V1
            flag.introducedIn.v1
                ? [flag.introducedIn.v1, renderValue(false), renderValue(flag.recommendedValue)]
                : ['(not in v1)', '', ''],
            // V2
            flag.introducedIn.v2
                ? [flag.introducedIn.v2, renderValue(flag.defaults?.v2 ?? false), renderValue(flag.recommendedValue)]
                : flag.defaults?.v2 !== undefined
                    ? ['(default in v2)', renderValue(flag.defaults?.v2), '']
                    : ['(not in v2)', '', ''],
        ]),
        ...oldBehavior(flag) ? [
            `**Compatibility with old behavior:** ${oldBehavior(flag)}`,
            '',
        ] : [],
        '',
    ]).join('\n');
}
function oldBehavior(flag) {
    switch (flag.type) {
        case flag_modeling_1.FlagType.ApiDefault: return flag.compatibilityWithOldBehaviorMd;
        case flag_modeling_1.FlagType.BugFix: return flag.compatibilityWithOldBehaviorMd;
        case flag_modeling_1.FlagType.VisibleContext: return undefined;
    }
}
function recommendedJson() {
    return [
        '```json',
        JSON.stringify({ context: feats.NEW_PROJECT_CONTEXT }, undefined, 2),
        '```',
    ].join('\n');
}
function v2flags() {
    return flags(flag => flag.introducedIn.v2 !== undefined);
}
function flags(pred) {
    const entries = Object.entries(feats.FLAGS)
        .filter(([_, flag]) => pred(flag));
    entries.sort((a, b) => firstCmp(
    // Sort by versions first
    (0, flag_modeling_1.compareVersions)(a[1].introducedIn.v2, b[1].introducedIn.v2), (0, flag_modeling_1.compareVersions)(a[1].introducedIn.v1, b[1].introducedIn.v1), 
    // Then sort by name
    a[0].localeCompare(b[0])));
    return entries;
}
function renderType(type) {
    switch (type) {
        case flag_modeling_1.FlagType.ApiDefault: return '(default)';
        case flag_modeling_1.FlagType.BugFix: return '(fix)';
        case flag_modeling_1.FlagType.VisibleContext: return '(config)';
    }
}
function renderTable(rows) {
    return [
        '',
        '| ' + rows[0].join(' | ') + ' |',
        '| ' + rows[0].map(_ => '-----').join(' | ') + ' |',
        ...rows.slice(1).map(row => '| ' + row.join(' | ') + ' |'),
        '',
    ].join('\n');
}
/**
 * Return the heading that will be used to caption this flag's details
 */
function flagDetailsHeading(name, _) {
    return name;
}
/**
 * Return a link that is valid on GitHub to refer to a heading
 */
function githubHeadingLink(heading) {
    return `#${heading.toLowerCase().replace(/ /g, '-').replace(/[^a-z0-9_-]/g, '')}`;
}
/**
 * Remove shared leading whitespace from all non-empty lines
 */
function dedent(body) {
    const lines = body.split('\n').filter((x) => x.trim() !== '');
    const leadingWs = lines.map(x => x.match(/^ */)?.[0].length ?? 0);
    const sharedWs = Math.min(...leadingWs);
    const re = new RegExp('^' + ' '.repeat(sharedWs), 'mg');
    return body.replace(re, '').trim();
}
function renderValue(x) {
    return `\`${JSON.stringify(x)}\``;
}
function renderLink(caption, link) {
    return `[${caption}](${link})`;
}
function mdEsc(x) {
    return x.replace(/_/g, '\\_');
}
async function updateMarkdownFile(filename, sections) {
    let contents = await fs_1.promises.readFile(filename, { encoding: 'utf-8' });
    for (const [section, value] of Object.entries(sections)) {
        const re = new RegExp(`<!-- BEGIN ${section} -->(.*)<!-- END ${section} -->`, 's');
        contents = contents.replace(re, `<!-- BEGIN ${section} -->\n${value}\n<!-- END ${section} -->`);
    }
    await fs_1.promises.writeFile(filename, contents, { encoding: 'utf-8' });
}
function firstCmp(...xs) {
    return xs.find(x => x !== 0) ?? 0;
}
main().catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exitCode = 1;
});
//# sourceMappingURL=data:application/json;base64,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