"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudArtifact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const metadata_1 = require("./metadata");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     *
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        try {
            jsiiDeprecationWarnings._aws_cdk_cx_api_CloudAssembly(assembly);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromManifest);
            }
            throw error;
        }
        // Implementation is defined in a separate file to break cyclic dependencies
        void (assembly), void (id), void (artifact);
        throw new Error('Implementation not overridden yet');
    }
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        try {
            jsiiDeprecationWarnings._aws_cdk_cx_api_CloudAssembly(assembly);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudArtifact);
            }
            throw error;
        }
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.tryGetArtifact(id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
    /**
     * An identifier that shows where this artifact is located in the tree
     * of nested assemblies, based on their manifests. Defaults to the normal
     * id. Should only be used in user interfaces.
     */
    get hierarchicalId() {
        return this.manifest.displayName ?? this.id;
    }
}
exports.CloudArtifact = CloudArtifact;
_a = JSII_RTTI_SYMBOL_1;
CloudArtifact[_a] = { fqn: "@aws-cdk/cx-api.CloudArtifact", version: "2.112.0" };
//# sourceMappingURL=data:application/json;base64,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