from pydantic import BaseModel, Field, model_validator
from typing import Annotated, Self, Type
from maleo.enums.status import (
    DataStatus,
    SimpleDataStatusMixin,
)
from maleo.schemas.mixins.identity import DataIdentifier
from maleo.schemas.mixins.timestamp import LifecycleTimestamp
from maleo.types.string import OptStr, OptListOfStrs
from ..enums.parameter import (
    ParameterGroup,
    ValueType as ValueTypeEnum,
)
from ..mixins.parameter import (
    _validate_value_type_and_options,
    Group,
    IsMandatory,
    Name,
    Aliases,
    ValueType,
    Options,
    IsNullable,
    Unit,
)
from ..mixins.rule import RuleData


class StandardParameterSchema(
    Unit[OptStr],
    IsNullable[bool],
    Options[OptListOfStrs],
    ValueType[ValueTypeEnum],
    Aliases[OptListOfStrs],
    Name[str],
    Group[ParameterGroup],
    IsMandatory[bool],
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    @model_validator(mode="after")
    def validate_value_type_and_options(self) -> Self:
        _validate_value_type_and_options(self.value_type, self.options)
        return self


class StandardRuleSchema(
    RuleData,
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


ListOfStandardRuleSchemas = list[StandardRuleSchema]


class StandardRuleSchemasMixin(BaseModel):
    rules: Annotated[
        ListOfStandardRuleSchemas, Field(ListOfStandardRuleSchemas, description="Rules")
    ] = ListOfStandardRuleSchemas()


class FullParameterSchema(
    StandardRuleSchemasMixin,
    StandardParameterSchema,
):
    pass


AnyParameterSchemaType = Type[StandardParameterSchema] | Type[FullParameterSchema]
AnyParameterSchema = StandardParameterSchema | FullParameterSchema


class FullRuleSchema(
    RuleData,
    StandardParameterSchema,
    SimpleDataStatusMixin[DataStatus],
    LifecycleTimestamp,
    DataIdentifier,
):
    pass


AnyRuleSchemaType = Type[StandardRuleSchema] | Type[FullRuleSchema]
AnyRuleSchema = StandardRuleSchema | FullRuleSchema
