# envshield/core/schema_manager.py
import datetime
from rich.console import Console
from rich.table import Table

from envshield.config import manager as config_manager
from envshield.parsers.factory import get_parser

console = Console()


def check_schema(file_path: str):
    """
    Validates a local environment file against the env.schema.toml,
    intelligently handling variables with default values.
    """
    console.print(
        f"\n[bold]Validating [magenta]{file_path}[/magenta] against schema...[/bold]"
    )

    # Load the schema and the local .env file
    schema = config_manager.load_schema()
    parser = get_parser(file_path)

    if not parser:
        console.print(f"[red]Error:[/red] No parser found for file type '{file_path}'.")
        return

    try:
        local_vars = parser.get_vars(file_path)
    except FileNotFoundError:
        console.print(f"[red]Error:[/red] File not found: '{file_path}'.")
        return

    # Get all variables defined in the schema
    schema_vars_all = set(schema.keys())

    # A variable is only considered "required" in the local file if it's in the schema
    # AND it does NOT have a defaultValue.
    schema_vars_required = {
        key for key, details in schema.items() if "defaultValue" not in details
    }

    # Determine what's actually missing and what's extra
    missing_in_local = schema_vars_required - local_vars
    extra_in_local = local_vars - schema_vars_all

    if not missing_in_local and not extra_in_local:
        console.print(
            "[bold green]✓ Your configuration is perfectly in sync with the schema![/bold green]"
        )
        return

    # If there are issues, build and display a report table
    table = Table(show_header=True, header_style="bold blue")
    table.add_column("Status", style="cyan")
    table.add_column("Variable Name", style="white")
    table.add_column("Source", style="white")

    for var in sorted(list(missing_in_local)):
        table.add_row("[red]Missing in Local[/red]", var, "env.schema.toml (Required)")

    for var in sorted(list(extra_in_local)):
        table.add_row("[yellow]Extra in Local[/yellow]", var, file_path)

    console.print(table)
    console.print(
        "\n[bold]Suggestion:[/bold] Please update your local file to match the schema contract."
    )


def sync_schema():
    """
    Generates a .env.example from the env.schema.toml.
    """
    console.print("\n[bold]Generating [cyan].env.example[/cyan] from schema...[/bold]")
    schema = config_manager.load_schema()
    output_file = ".env.example"

    header = (
        f"# This file was auto-generated by EnvShield on {datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n"
        f"# It is generated from the contract defined in env.schema.toml.\n"
        f"# DO NOT EDIT THIS FILE MANUALLY.\n\n"
    )

    content = header
    for key, details in schema.items():
        description = details.get("description")
        if description:
            content += f"# {description}\n"

        default_value = details.get("defaultValue", "")
        content += f"{key}={default_value}\n\n"

    try:
        with open(output_file, "w") as f:
            f.write(content)
        console.print(
            f"[bold green]✓[/bold green] Successfully created/updated [bold cyan]{output_file}[/bold cyan]!"
        )
    except IOError as e:
        console.print(f"[red]Error:[/red] Could not write to {output_file}: {e}")
