# envshield/core/setup_manager.py
# Contains the core business logic for the 'setup' command.

import datetime
import os
import re
from typing import Dict, List

import questionary
from rich.console import Console
from rich.panel import Panel

from .exceptions import EnvShieldException
from rich.prompt import Prompt

console = Console()
EXAMPLE_FILE = ".env.example"
OUTPUT_FILE = ".env"


def _is_secret_key(key: str) -> bool:
    """A simple heuristic to determine if a key is likely a secret."""
    key_lower = key.lower()
    return any(
        keyword in key_lower for keyword in ["secret", "token", "key", "password"]
    )


def run_setup():
    """
    Guides a user through creating their local .env file from .env.example.
    """
    console.print(
        Panel(
            f"[bold cyan]Welcome to EnvShield Setup[/bold cyan]\n\nThis wizard will help you create your local [magenta]{OUTPUT_FILE}[/magenta] file.",
            title="✨ Local Setup ✨",
            border_style="green",
        )
    )

    # Step 1: Check for prerequisites
    if not os.path.exists(EXAMPLE_FILE):
        raise EnvShieldException(
            f"'{EXAMPLE_FILE}' not found. Please run 'envshield schema sync' first to generate it."
        )

    if os.path.exists(OUTPUT_FILE):
        overwrite = questionary.confirm(
            f"A '{OUTPUT_FILE}' file already exists. Do you want to overwrite it?",
            default=False,
        ).ask()
        if not overwrite:
            console.print("[yellow]Setup cancelled.[/yellow]")
            return

    # Step 2: Parse the example file
    console.print(
        f"\n[bold]Reading variables from [cyan]{EXAMPLE_FILE}[/cyan]...[/bold]"
    )
    vars_to_prompt: List[str] = []
    final_vars: Dict[str, str] = {}

    try:
        with open(EXAMPLE_FILE, "r") as f:
            for line in f:
                line = line.strip()
                if not line or line.startswith("#"):
                    continue
                if "=" in line:
                    key, value = line.split("=", 1)
                    key = key.strip()
                    value = value.strip()
                    if not value:
                        vars_to_prompt.append(key)
                    else:
                        final_vars[key] = value
    except IOError as e:
        raise EnvShieldException(f"Could not read '{EXAMPLE_FILE}': {e}")

    # Step 3: Prompt for empty variables
    if not vars_to_prompt:
        console.print("[green]✓ No empty variables found to configure.[/green]")
    else:
        console.print(
            "\n[bold]Please provide values for the following variables:[/bold]"
        )
        for key in vars_to_prompt:
            new_value = Prompt.ask(
                f"  Please enter the value for [bold cyan]{key}[/bold cyan]",
                password=_is_secret_key(key),
            )
            final_vars[key] = new_value

    # Step 4: Write the new .env file
    try:
        with open(OUTPUT_FILE, "w") as f:
            f.write(
                f"# Auto-generated by 'envshield setup' on {datetime.datetime.now().strftime('%Y-%m-%d')}\n\n"
            )
            for key, value in final_vars.items():
                # More robustly quote values that contain special characters or spaces.
                if (
                    re.search(r"[#\s=]", value)
                    and not (value.startswith("'") and value.endswith("'"))
                    and not (value.startswith('"') and value.endswith('"'))
                ):
                    f.write(f'{key}="{value}"\n')
                else:
                    f.write(f"{key}={value}\n")
        console.print(
            f"\n[bold green]✓ Successfully created your [magenta]{OUTPUT_FILE}[/magenta] file![/bold green]"
        )
    except IOError as e:
        raise EnvShieldException(f"Could not write to '{OUTPUT_FILE}': {e}")
