# envshield/tests/core/test_schema_manager.py
from typer.testing import CliRunner

import os
from envshield.core import schema_manager

runner = CliRunner()


def test_check_schema_in_sync(mocker, tmp_path):
    """Tests the check command when the local file is in sync with the schema."""
    with runner.isolated_filesystem(temp_dir=tmp_path):
        mocker.patch(
            "envshield.config.manager.load_schema",
            return_value={"DATABASE_URL": {"description": "DB URL", "secret": True}},
        )

        with open(".env.local", "w") as f:
            f.write("DATABASE_URL=postgres://user:pass@localhost/db\n")

        mock_console = mocker.patch("envshield.core.schema_manager.console")

        schema_manager.check_schema(".env.local")

        mock_console.print.assert_any_call(
            "[bold green]✓ Your configuration is perfectly in sync with the schema![/bold green]"
        )


def test_check_schema_out_of_sync(mocker, tmp_path):
    """Tests the check command when the local file is out of sync."""
    with runner.isolated_filesystem(temp_dir=tmp_path):
        mocker.patch(
            "envshield.config.manager.load_schema",
            return_value={"DATABASE_URL": {}, "NEW_VARIABLE": {}},
        )

        with open(".env.local", "w") as f:
            f.write("DATABASE_URL=some_value\nOLD_VARIABLE=true")

        mock_console = mocker.patch("envshield.core.schema_manager.console")

        schema_manager.check_schema(".env.local")

        assert mock_console.print.call_count > 2


def test_sync_schema_generates_perfect_file(mocker, tmp_path):
    """Tests that schema sync correctly generates a .env.example file with exact content."""
    with runner.isolated_filesystem(temp_dir=tmp_path) as td:
        schema_data = {
            "DATABASE_URL": {"description": "DB URL", "secret": True},
            "LOG_LEVEL": {"description": "Log verbosity", "defaultValue": "info"},
        }
        mocker.patch("envshield.config.manager.load_schema", return_value=schema_data)

        output_file = os.path.join(td, ".env.example")

        schema_manager.sync_schema()

        assert os.path.exists(output_file)
        with open(output_file, "r") as f:
            content = f.read()

        expected_content = (
            "# It is generated from the contract defined in env.schema.toml.\n"
            "# DO NOT EDIT THIS FILE MANUALLY.\n\n"
            "# DB URL\n"
            "DATABASE_URL=\n\n"
            "# Log verbosity\n"
            "LOG_LEVEL=info\n\n"
        )
        assert expected_content in content
