# ivcap_client: Python library to interact with an IVCAP Cluster

A python library to interact with the published API of IVCAP. Almost
all the code of this library is auto generated by [openapi-python-client](https://github.com/openapi-generators/openapi-python-client) from [IVCAP's](https://github.com/reinventingscience/ivcap-core/) `openapi3.json` file.

> CAUTION: This README is out of date, for the moment it is better to check out
> the examples in [examples](./examples). We'll try to fix this soon.

## Usage

* [Create a client](#create)
* [Call a Service](#call)
* [Use Models](#use)
* [Example: Order a Service](#order)
* [Things to know](#know)

### Create a client <a name="create"></a>

```python
from ivcap_client import AuthenticatedClient

client = AuthenticatedClient(base_url="https://api.ivcap.net", token="JWT")
```

> Currently, the best way to get the token is to use the [ivcap cli](https://github.com/reinventingscience/ivcp-cli) and execute `ivcap context get access-token`. Obviously,
this assumes that the current context of `ivcap` is the same as the one the above
client is for.

By default, when you're calling an HTTPS API it will attempt to verify that SSL is working correctly. Using certificate verification is highly recommended most of the time, but sometimes you may need to authenticate to a server (especially an internal server) using a custom certificate bundle.

```python
client = AuthenticatedClient(
    base_url="https://internal_api.example.com",
    token="SuperSecretToken",
    verify_ssl="/path/to/certificate_bundle.pem",
)
```

You can also disable certificate validation altogether, but beware that **this is a security risk**.

```python
client = AuthenticatedClient(
    base_url="https://internal_api.example.com",
    token="SuperSecretToken",
    verify_ssl=False
)
```

### Call a Service and use Models<a name="call"></a>

```python
from ivcap_client.models import ServiceListRT
from ivcap_client.api.service import service_list
from ivcap_client.types import Response

services: ServiceListRT = service_list.sync(client=client, limit=1)
# or if you need more info (e.g. status_code)
response: Response[ServiceListRT] = service_list.sync_detailed(client=client, limit)
```

Or do the same thing with an async version:

```python
from ivcap_client.models import ServiceListRT
from ivcap_client.api.service import service_list
from ivcap_client.types import Response

services: ServiceListRT = await service_list.asyncio(client=client, limit=2)
# or if you need more info (e.g. status_code)
response: Response[ServiceListRT] = await service_list.asyncio_detailed(client=client, limit)
```

### Use Models<a name="use"></a>

The above call returns a `ServiceListRT` object. To find out what attributes that contains:

```python
>>> services.to_dict().keys()
dict_keys(['at-time', 'links', 'services'])
```

and to pretty-print the list of services:

```python
import pprint
pp = pprint.PrettyPrinter(indent=2)

>>> pp.pprint(list(map(lambda el: el.to_dict(), services.services)))
[ { 'id': 'urn:ivcap:service:8773f79e-d46c-559a-a63c-54c4e2a9d9a1',
    'links': { ... },
    'name': 'infer-with-paddle-paddle',
    'provider': { 'id': 'urn:ivcap:provider:4c65b865-df6a-4977-982a-f96b19c1fda0'}},
  { 'id': 'urn:ivcap:service:85f4586e-af1e-5200-94ba-0be8651740ed',
    'links': { ... },
    'name': 'Gradient Text Image',
    'provider': { 'id': 'urn:ivcap:provider:4c65b865-df6a-4977-982a-f96b19c1fda0'}
}]
```

### Example: Order a Service <a name="order"></a>

To get the list of parameters for a particular service, using the list of services we obtained above:

```python
from ivcap_client.api.service import service_read
from ivcap_client.models import ServiceStatusRT

exampleService:ServiceStatusRT = service_read.sync(client=client, id=services.services[0].id)
>>> exampleService.to_dict().keys()
dict_keys(['id', 'links', 'parameters', 'account', 'description', 'metadata', 'name', 'provider', 'tags'])
>>> exampleService.description
'Creates an image with a customizable text.'
>>> pp.pprint(list(map(lambda el: el.to_dict(), exampleService.parameters)))
[ { 'description': '',
    'label': 'Message to display',
    'name': 'msg',
    'optional': False,
    'options': [],
    'type': 'string',
    'unit': ''},
  { 'description': '',
    'label': 'Image artifact to use as background',
    'name': 'img-art',
    'optional': True,
    'options': [],
    'type': 'artifact',
    'unit': ''},
  { 'description': '',
    'label': 'Image url (external) to use as background',
    'name': 'img-url',
    'optional': True,
    'options': [],
    'type': 'string',
    'unit': ''}]
```

To order a service:

```python
from typing import List
from ivcap_client.api.order import order_create
from ivcap_client.models import OrderRequestT, ParameterT, OrderRequestT, OrderStatusRT

img_url = 'https://juststickers.in/wp-content/uploads/2016/07/go-programming-language.png'
service_id = 'urn:ivcap:service:85f4586e-af1e-5200-94ba-0be8651740ed'
account_id = os.environ['IVCAP_ACCOUNT_ID']

p = [
    ParameterT(name='msg', value='Hello World'),
    ParameterT(name='img-url', value=img_url)
]
req = OrderRequestT(name='test 1', parameters=p, service_id=service_id , account_id=account_id)
order_resp = order_create.sync_detailed(client=client, json_body=req)
order:OrderStatusRT = order_resp.parsed
if not order:
    raise Exception(f'Order request failed: ${order_resp.status_code}')

>>> pp.pprint(order.to_dict())
{ 'account': {'id': 'urn:ivcap:account:4c65b865-df6a-4977-982a-f96b19c1fda0'},
  'id': 'urn:ivcap:order:cd04597a-2d55-4f1c-906c-995450f30599',
  'links': { 'self': 'http://localhost:8080/1/orders/cd04597a-2d55-4f1c-906c-995450f30599'},
  'name': 'test 1',
  'ordered_at': '2023-05-03T16:31:04+10:00',
  'parameters': [ {'name': 'msg', 'value': 'Hello World'},
                  { 'name': 'img-url',
                    'value': 'https://juststickers.in/wp-content/uploads/2016/07/go-programming-language.png'}],
  'products': [],
  'service': { 'id': 'urn:ivcap:service:85f4586e-af1e-5200-94ba-0be8651740ed',
               'links': { 'self': 'http://localhost:8080/1/services/85f4586e-af1e-5200-94ba-0be8651740ed'}},
  'status': 'pending'}

```

Later on we can check on the order:

```python
from ivcap_client.api.order import order_read

ostatus = order_read.sync(client=client, id=order.id)
>>> pp.pprint(ostatus.to_dict())
{ 'account': {'id': 'urn:ivcap:account:4c65b865-df6a-4977-982a-f96b19c1fda0'},
  'finished_at': '2023-05-04T00:29:19Z',
  'id': 'urn:ivcap:order:de15958e-9575-444a-aa25-69ff259e2345',
  'links': { ... },
  'name': 'test 1',
  'ordered_at': '2023-05-04T00:28:29Z',
  'parameters': [ {'name': 'msg', 'value': 'Hello World'},
                  { 'name': 'img-url',
                    'value': 'https://juststickers.in/wp-content/uploads/2016/07/go-programming-language.png'}],
  'products': [ { 'id': 'urn:ivcap:artifact:eae6aceb-aec5-4eec-a918-bd4575891168',
                  'links': { ... },
                  'mime-type': 'image/png',
                  'name': 'out.png',
                  'size': 0}],
  'service': { 'id': 'urn:ivcap:service:85f4586e-af1e-5200-94ba-0be8651740ed',
               'links': { ... }},
  'started_at': '2023-05-04T00:28:45Z',
  'status': 'succeeded'}

```

### Things to know<a name="know"></a>

1. Every path/method combo becomes a Python module with four functions:
    1. `sync`: Blocking request that returns parsed data (if successful) or `None`
    1. `sync_detailed`: Blocking request that always returns a `Request`, optionally with `parsed` set if the request was successful.
    1. `asyncio`: Like `sync` but async instead of blocking
    1. `asyncio_detailed`: Like `sync_detailed` but async instead of blocking

1. All path/query params, and bodies become method arguments.
1. If your endpoint had any tags on it, the first tag will be used as a module name for the function (my_tag above)
1. Any endpoint which did not have a tag will be in `ivcap_client.api.default`
