from http import HTTPStatus
from typing import Any, Optional, Union, cast

import httpx

from ... import errors
from ...client import AuthenticatedClient, Client
from ...models.artifact_status_rt import ArtifactStatusRT
from ...models.bad_request_t import BadRequestT
from ...models.invalid_scopes_t import InvalidScopesT
from ...models.not_implemented_t import NotImplementedT
from ...models.resource_not_found_t import ResourceNotFoundT
from ...types import Response


def _get_kwargs(
    id: str,
) -> dict[str, Any]:
    _kwargs: dict[str, Any] = {
        "method": "get",
        "url": f"/1/artifacts/{id}",
    }

    return _kwargs


def _parse_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Optional[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]:
    if response.status_code == 200:
        response_200 = ArtifactStatusRT.from_dict(response.json())

        return response_200
    if response.status_code == 400:
        response_400 = BadRequestT.from_dict(response.json())

        return response_400
    if response.status_code == 401:
        response_401 = cast(Any, None)
        return response_401
    if response.status_code == 403:
        response_403 = InvalidScopesT.from_dict(response.json())

        return response_403
    if response.status_code == 404:
        response_404 = ResourceNotFoundT.from_dict(response.json())

        return response_404
    if response.status_code == 501:
        response_501 = NotImplementedT.from_dict(response.json())

        return response_501
    if response.status_code == 503:
        response_503 = cast(Any, None)
        return response_503
    if client.raise_on_unexpected_status:
        raise errors.UnexpectedStatus(response.status_code, response.content)
    else:
        return None


def _build_response(
    *, client: Union[AuthenticatedClient, Client], response: httpx.Response
) -> Response[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]:
    return Response(
        status_code=HTTPStatus(response.status_code),
        content=response.content,
        headers=response.headers,
        parsed=_parse_response(client=client, response=response),
    )


def sync_detailed(
    id: str,
    *,
    client: AuthenticatedClient,
) -> Response[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]:
    """read artifact

     Show artifacts by ID

    Args:
        id (str): ID of artifacts to show Example: urn:ivcap:artifacts.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]
    """

    kwargs = _get_kwargs(
        id=id,
    )

    response = client.get_httpx_client().request(
        **kwargs,
    )

    return _build_response(client=client, response=response)


def sync(
    id: str,
    *,
    client: AuthenticatedClient,
) -> Optional[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]:
    """read artifact

     Show artifacts by ID

    Args:
        id (str): ID of artifacts to show Example: urn:ivcap:artifacts.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]
    """

    return sync_detailed(
        id=id,
        client=client,
    ).parsed


async def asyncio_detailed(
    id: str,
    *,
    client: AuthenticatedClient,
) -> Response[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]:
    """read artifact

     Show artifacts by ID

    Args:
        id (str): ID of artifacts to show Example: urn:ivcap:artifacts.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Response[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]
    """

    kwargs = _get_kwargs(
        id=id,
    )

    response = await client.get_async_httpx_client().request(**kwargs)

    return _build_response(client=client, response=response)


async def asyncio(
    id: str,
    *,
    client: AuthenticatedClient,
) -> Optional[Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]]:
    """read artifact

     Show artifacts by ID

    Args:
        id (str): ID of artifacts to show Example: urn:ivcap:artifacts.

    Raises:
        errors.UnexpectedStatus: If the server returns an undocumented status code and Client.raise_on_unexpected_status is True.
        httpx.TimeoutException: If the request takes longer than Client.timeout.

    Returns:
        Union[Any, ArtifactStatusRT, BadRequestT, InvalidScopesT, NotImplementedT, ResourceNotFoundT]
    """

    return (
        await asyncio_detailed(
            id=id,
            client=client,
        )
    ).parsed
