"""The building blocks of energy-based Associative Memory"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core.qmd.

# %% auto 0
__all__ = ['NeuronLayer', 'legendre_transform', 'LinearSynapse', 'LinearSynapseWithBias', 'BiasSynapse', 'ConvSynapse', 'HAM',
           'VectorizedHAM']

# %% ../nbs/00_core.qmd 2
from typing import *
import equinox as eqx
import jax, jax.numpy as jnp, jax.random as jr, jax.tree_util as jtu
from fastcore.basics import *
import functools as ft

# %% ../nbs/00_core.qmd 5
class NeuronLayer(eqx.Module):
    """Neuron layers represent dynamic variables that evolve during inference (i.e., memory retrieval/error correction)"""
    lagrangian: Callable # The scalar-valued Lagrangian function:  x |-> R
    shape: Tuple[int] # The shape of the neuron layer

# %% ../nbs/00_core.qmd 7
@patch
def activations(self: NeuronLayer, x): 
    """Use autograd to compute the activations of the neuron layer from the Lagrangian"""
    return jax.grad(self.lagrangian)(x)

# %% ../nbs/00_core.qmd 12
@patch
def init(self: NeuronLayer, bs: Optional[int] = None):
    """Return an empty initial neuron state"""
    if bs is None or bs == 0: return jnp.zeros(self.shape) # No batch dimension
    return jnp.zeros((bs, *self.shape))

# %% ../nbs/00_core.qmd 16
def legendre_transform(
    F: Callable # The function to transform
    ):
    "Transform scalar F(x) into the dual Fhat(xhat, x) using the Legendre transform"

    # We define custom gradient rules to give jax some autograd shortcuts
    @jax.custom_jvp
    def Fhat(xhat, x): return jnp.multiply(xhat, x).sum() - F(x)

    @Fhat.defjvp
    def Fhat_jvp(primals, tangents):
        (xhat, x), (dxhat, dx) = primals, tangents
        o, do = Fhat(xhat, x), jnp.multiply(x, dxhat).sum()
        return o, do

    return Fhat

# %% ../nbs/00_core.qmd 20
@patch
def energy(self: NeuronLayer, xhat, x): 
    """The energy of the neuron layer is the Legendre transform of the Lagrangian"""
    return legendre_transform(self.lagrangian)(xhat, x)

# %% ../nbs/00_core.qmd 24
NeuronLayer.sigma = NeuronLayer.activations
NeuronLayer.E = NeuronLayer.energy

# %% ../nbs/00_core.qmd 26
@patch
def __repr__(self: NeuronLayer): 
    """Look nice when inspected"""
    return f"NeuronLayer(lagrangian={self.lagrangian}, shape={self.shape})"

@patch
def __post_init__(self: NeuronLayer): 
    """Ensure the neuron shape is a tuple"""
    print("Post init")
    if isinstance(self.shape, int): self.shape = (self.shape,)

# %% ../nbs/00_core.qmd 28
class LinearSynapse(eqx.Module):
    """The energy synapse corrolary of the linear layer in standard neural networks"""
    W: jax.Array
    def __call__(self, xhat1:jax.Array, xhat2:jax.Array):
        "Compute the interaction energy between activations `xhat1` and `xhat2`."
        # Best to use batch-dim agnostic operations
        return -jnp.einsum("...c,...d,cd->...", xhat1, xhat2, self.W)

    @classmethod
    def rand_init(cls, key: jax.Array, x1_dim: int, x2_dim: int):
        Winit = 0.02 * jr.normal(key, (x1_dim, x2_dim))
        return cls(W=Winit)

# %% ../nbs/00_core.qmd 32
class LinearSynapseWithBias(eqx.Module):
    """A linear synapse with a bias"""
    W: jax.Array
    b: jax.Array
    def __call__(self, xhat1:jax.Array, xhat2:jax.Array):
        "Compute the interaction energy between activations `xhat1` and `xhat2`."
        return -jnp.einsum("...c,...d,cd->...", xhat1, xhat2, self.W) - jnp.multiply(xhat2, self.b).sum()

    @classmethod
    def rand_init(cls, key: jax.Array, x1_dim: int, x2_dim: int):
        Winit = 0.02 * jr.normal(key, (x1_dim, x2_dim))
        binit = 0.02 * jr.normal(key, (x2_dim,))
        return cls(W=Winit, b=binit)

class BiasSynapse(eqx.Module):
    """Energy defines constant input to a neuron layer"""
    b: jax.Array
    def __call__(self, xhat:jax.Array):
        "Compute the interaction energy between activations `xhat` and the bias `b`."
        return -jnp.einsum("...c,c->...", xhat, self.b)

# %% ../nbs/00_core.qmd 35
class ConvSynapse(eqx.Module):
    """The energy corrolary of a convolutional layer in standard neural networks"""
    W: jax.Array
    window_strides: Tuple[int, int]
    padding: str

    @classmethod
    def from_conv_params(cls,
                         key, # Random weight generation seed
                         channels_out:int, # nfilters
                         channels_in:int, # Channels in input
                         filter_shape:Tuple[int,int], # (h,w) size of filter
                         window_strides:Tuple[int,int], # (h, w) size of each step
                         padding="SAME",
                         mu=0.5, # Mean of W initialization
                         sigma=0.01, # stdev of W initialization
                        ):
        "Initialize the ConvSynapse from the convolution parameters"
        W = jr.normal(key, (channels_out, channels_in, *filter_shape)) * sigma + mu
        return cls(W=W, window_strides=window_strides, padding=padding)

    def forward_conv(self, xhat1):
        return jax.lax.conv_general_dilated(xhat1, self.W, window_strides=self.window_strides, padding=self.padding, dimension_numbers=("NHWC", "OIHW", "NHWC")) # in (1, H2, W2, Cout)
        
    def __call__(self, xhat1, xhat2):
        """Energy of convolution is defined only for a single input"""
        assert len(xhat1.shape)== 3, f"Shape should be (H,W,C), no batch dimension. Got `{xhat1.shape}` for xhat1"
        assert len(xhat2.shape)== 3, f"Shape should be (H,W,C), no batch dimension. Got `{xhat2.shape}` for xhat2"
     
        xhat1 = xhat1[None] # Add batch dimension
        xhat2 = xhat2[None] # Add batch dimension
        
        sig1_into_2 = self.forward_conv(xhat1)
        return -jnp.multiply(xhat2, sig1_into_2).sum()

# %% ../nbs/00_core.qmd 39
class HAM(eqx.Module): 
    "A Hypergraph wrapper connecting all dynamic states (neurons) and learnable parameters (synapses) for our associative memory" 
    neurons: Dict[str, NeuronLayer]
    hypersynapses: Dict[str, eqx.Module] 
    connections: List[Tuple[Tuple, str]]

# %% ../nbs/00_core.qmd 43
@patch(as_prop=True)
def n_neurons(self:HAM) -> int:
   "Total number of neurons"
   return len(self.neurons)

@patch(as_prop=True)
def n_hypersynapses(self:HAM) -> int:
   "Total number of hypersynapses"
   return len(self.hypersynapses)

@patch(as_prop=True)
def n_connections(self:HAM) -> int:
   "Total number of connections"
   return len(self.connections)

# %% ../nbs/00_core.qmd 44
@patch
def init_states(self: HAM, bs: Optional[int] = None):
    """Initialize all neuron states"""
    if bs is not None and bs > 0: warn("Vectorize with `ham.vectorize()` before processing batched states")
    xs = {k: v.init(bs) for k, v in self.neurons.items()}
    return xs

# %% ../nbs/00_core.qmd 50
@patch
def activations(self: HAM, xs):
    """Convert hidden states of each neuron into their activations"""
    xhats = {k: v.sigma(xs[k]) for k, v in self.neurons.items()}
    return xhats

# %% ../nbs/00_core.qmd 53
@patch
def neuron_energies(self: HAM, xhats, xs):
    """Retrieve the energies of each neuron in the HAM"""
    return {k: self.neurons[k].energy(xhats[k], xs[k]) for k in self.neurons.keys()}

@patch
def connection_energies(self: HAM, xhats):
    """Get the energy for each connection"""
    def get_energy(neuron_set, s):
        neighbor_xhats = [xhats[k] for k in neuron_set]
        return self.hypersynapses[s](*neighbor_xhats)
    return [get_energy(neuron_set, s) for neuron_set, s in self.connections]

@patch
def energy_tree(self: HAM, xhats, xs):
    """Return energies for each individual component"""
    neuron_energies = self.neuron_energies(xhats, xs)
    connection_energies = self.connection_energies(xhats)
    return {"neurons": neuron_energies, "connections": connection_energies}

@patch
def energy(self: HAM, xhats, xs):
    """The complete energy of the HAM"""
    energy_tree = self.energy_tree(xhats, xs)
    return jtu.tree_reduce(lambda E, acc: acc + E, energy_tree, 0)

# %% ../nbs/00_core.qmd 57
@patch
def dEdact(self: HAM, xhats, xs, return_energy=False):
    """Calculate gradient of system energy w.r.t. each activation"""
    if return_energy: return jax.value_and_grad(self.energy)(xhats, xs)
    return jax.grad(self.energy)(xhats, xs)

# %% ../nbs/00_core.qmd 59
def _docstring_from(source_func):
    """Decorator that copies the docstring from source_func to the decorated function"""
    def decorator(target_func):
        if source_func.__doc__: target_func.__doc__ = source_func.__doc__
        return target_func
    return decorator

# %% ../nbs/00_core.qmd 60
class VectorizedHAM(eqx.Module):
    """Re-expose HAM API with vectorized inputs. No new HAM behaviors should be implemented in this class."""
    _ham: eqx.Module
    @property
    def neurons(self): return self._ham.neurons
    @property
    def hypersynapses(self): return self._ham.hypersynapses
    @property
    def connections(self): return self._ham.connections
    @property
    def n_neurons(self): return self._ham.n_neurons
    @property
    def n_hypersynapses(self): return self._ham.n_hypersynapses
    @property
    def n_connections(self): return self._ham.n_connections

    @property
    def _batch_axes(self: HAM):
        """A helper function to tell vmap to batch along the 0'th dimension of each state in the HAM."""
        return {k: 0 for k in self._ham.neurons.keys()}

    @_docstring_from(HAM.init_states)
    def init_states(self, bs=None): 
        if bs is None or bs == 0: warn("Vectorized HAMs should be initialized with a batch size")
        xs = {k: v.init(bs) for k, v in self.neurons.items()}
        return xs

    @_docstring_from(HAM.activations)
    def activations(self, xs): return jax.vmap(self._ham.activations, in_axes=(self._batch_axes,))(xs)

    @_docstring_from(HAM.connection_energies)
    def connection_energies(self, xhats): return jax.vmap(self._ham.connection_energies, in_axes=(self._batch_axes,))(xhats)

    @_docstring_from(HAM.neuron_energies)
    def neuron_energies(self, xhats, xs): return jax.vmap(self._ham.neuron_energies, in_axes=(self._batch_axes, self._batch_axes))(xhats, xs)

    @_docstring_from(HAM.energy_tree)
    def energy_tree(self, xhats, xs): return jax.vmap(self._ham.energy_tree, in_axes=(self._batch_axes, self._batch_axes))(xhats, xs)

    @_docstring_from(HAM.energy)
    def energy(self, xhats, xs): return jax.vmap(self._ham.energy, in_axes=(self._batch_axes, self._batch_axes))(xhats, xs)

    @_docstring_from(HAM.dEdact)
    def dEdact(self, xhats, xs, return_energy=False): return jax.vmap(self._ham.dEdact, in_axes=(self._batch_axes, self._batch_axes, None))(xhats, xs, return_energy)

    def unvectorize(self): return self._ham
    def vectorize(self): return self

# %% ../nbs/00_core.qmd 61
@patch
def vectorize(self: HAM):
    """Vectorize to work on batches of inputs"""
    return VectorizedHAM(self)

@patch
def unvectorize(self: HAM):
    """Unvectorize to work on single inputs"""
    return self
