"""The well-behaved dynamics of associative memories is described by the Lagrangians of the neurons."""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_lagrangians.qmd.

# %% auto 0
__all__ = ['Scalar', 'Tensor', 'lagr_identity', 'lagr_repu', 'lagr_relu', 'lagr_exp', 'lagr_rexp', 'lagr_tanh', 'lagr_sigmoid',
           'lagr_softmax', 'lagr_layernorm', 'lagr_spherical_norm', 'lagr_ghostmax', 'ghostmax']

# %% ../nbs/01_lagrangians.qmd 3
import equinox as eqx
from typing import Union, Callable, Tuple, Dict, List, Optional
import jax
import jax.numpy as jnp
import jax.tree_util as jtu
import jax.random as jr
from jax import lax
from jax._src.numpy.reductions import _reduction_dims
from jax._src.numpy.util import promote_args_inexact
import numpy as np
from jaxtyping import Float, Array

Scalar = Float[Array, ""]
Tensor = Float[Array, "..."]

# %% ../nbs/01_lagrangians.qmd 5
def lagr_identity(x: Array, # Input tensor
                  ) -> Float: # Output scalar
    """The Lagrangian whose activation function is simply the identity."""
    return 0.5 * jnp.power(x, 2).sum()

# %% ../nbs/01_lagrangians.qmd 8
def _repu(x: Array, # Input tensor
          n: float # Degree of the polynomial in the power unit
          ) -> Float: # Output scalar
    return jnp.maximum(x, 0) ** n


def lagr_repu(x: Array, # Input tensor
              n: float # Degree of the polynomial in the power unit
              ) -> Float: # Output scalar
    """Rectified Power Unit of degree `n`"""
    return 1 / n * jnp.power(jnp.maximum(x, 0), n).sum()

# %% ../nbs/01_lagrangians.qmd 11
def lagr_relu(x: Array, # Input tensor
              ) -> Float: # Output scalar
    """Rectified Linear Unit. Same as `lagr_repu` of degree 2"""
    return lagr_repu(x, 2)

# %% ../nbs/01_lagrangians.qmd 13
def lagr_exp(x: Array, # Input tensor
             beta: float = 1.0, # Inverse temperature
             ) -> Float: # Output scalar
    """Exponential activation function, as in [Demicirgil et al.](https://arxiv.org/abs/1702.01929). Operates elementwise"""
    return 1 / beta * jnp.exp(beta * x).sum()

# %% ../nbs/01_lagrangians.qmd 16
def _rexp(
    x: Array, # Input tensor
    beta: float = 1.0,  # Inverse temperature
) -> Float: # Output scalar
    """Rectified exponential activation function"""
    xclipped = jnp.maximum(x, 0)
    return jnp.exp(beta * xclipped) - 1


def lagr_rexp(x: Array,
              beta: float = 1.0, # Inverse temperature
              ) -> Float: # Output scalar
    """Lagrangian of the Rectified exponential activation function"""
    xclipped = jnp.maximum(x, 0)
    return (jnp.exp(beta * xclipped) / beta - xclipped).sum()

# %% ../nbs/01_lagrangians.qmd 17
@jax.custom_jvp
def _lagr_tanh(x: Array, # Input tensor
               beta: float = 1.0, # Inverse temperature
               ) -> Float: # Output scalar
    return 1 / beta * jnp.log(jnp.cosh(beta * x))


@_lagr_tanh.defjvp
def _lagr_tanh_defjvp(primals, tangents):
    x, beta = primals
    x_dot, beta_dot = tangents
    primal_out = _lagr_tanh(x, beta)
    tangent_out = jnp.tanh(beta * x) * x_dot
    return primal_out, tangent_out


def lagr_tanh(x: Array, # Input tensor
              beta: float = 1.0, # Inverse temperature
              ) -> Float: # Output scalar
    """Lagrangian of the tanh activation function"""
    return _lagr_tanh(x, beta).sum()

# %% ../nbs/01_lagrangians.qmd 19
@jax.custom_jvp
def _lagr_sigmoid(
    x: Array, # Input tensor
    beta: float = 1.0,  # Inverse temperature
    ) -> Float: # Output scalar
    """The lagrangian of a sigmoid that we can define custom JVPs of"""
    return 1. / beta * jnp.log(jnp.exp(beta * x) + 1)


def _sigmoid(
    x: Array, # Input tensor
    beta: float = 1.0,  # Inverse temperature
    ) -> Float: # Output scalar
    """The basic sigmoid"""
    return 1. / (1 + jnp.exp(-beta * x))


@_lagr_sigmoid.defjvp
def _lagr_sigmoid_jvp(primals, tangents):
    x, beta = primals
    x_dot, beta_dot = tangents
    primal_out = _lagr_sigmoid(x, beta)
    tangent_out = (
        _sigmoid(x, beta=beta) * x_dot
    )  # Manually defined sigmoid
    return primal_out, tangent_out


def lagr_sigmoid(
    x: Array, # Input tensor
    beta: float = 1.0,  # Inverse temperature
) -> Float: # Output scalar
    """The lagrangian of the sigmoid activation function"""
    return _lagr_sigmoid(x, beta=beta).sum()

# %% ../nbs/01_lagrangians.qmd 22
def lagr_softmax(
    x: Array, # Input tensor
    beta: float = 1.0,  # Inverse temperature
    axis: int = -1, # Dimension over which to apply logsumexp
) -> Float: # Output scalar
    """The lagrangian of the softmax -- the logsumexp"""
    return 1 / beta * jax.nn.logsumexp(beta * x, axis=axis, keepdims=False)

# %% ../nbs/01_lagrangians.qmd 25
def _simple_layernorm(
    x: Array, # Input tensor
    gamma: float = 1.0,  # Scale the stdev
    delta: Union[float, Array] = 0.0,  # Shift the mean
    axis: int = -1,  # Which axis to normalize
    eps: float = 1e-5,  # Prevent division by 0
) -> Array: # Layer normalized `x`
    """Layer norm activation function"""
    xmean = x.mean(axis, keepdims=True)
    xmeaned = x - xmean
    denominator = jnp.sqrt(jnp.power(xmeaned, 2).mean(axis, keepdims=True) + eps)
    return gamma * xmeaned / denominator + delta


def lagr_layernorm(
    x: Array, # Input tensor
    gamma: float = 1.0,  # Scale the stdev
    delta: Union[float, Array] = 0.0,  # Shift the mean
    axis: int = -1,  # Which axis to normalize
    eps: float = 1e-5,  # Prevent division by 0
) -> Float: # Output scalar
    """Lagrangian of the layer norm activation function. 
    
    `gamma` must be a float, not a vector.
    """
    D = x.shape[axis] if axis is not None else x.size
    xmean = x.mean(axis, keepdims=True)
    xmeaned = x - xmean
    y = jnp.sqrt(jnp.power(xmeaned, 2).mean(axis, keepdims=True) + eps)
    return (D * gamma * y + (delta * x).sum()).sum()

# %% ../nbs/01_lagrangians.qmd 27
def _simple_spherical_norm(
    x: Array, # Input tensor
    gamma: float = 1.0,  # Scale the stdev
    delta: Union[float, Array] = 0.0,  # Shift the mean
    axis: int = -1,  # Which axis to normalize
    eps=1e-5,  # Prevent division by 0
):
    """Spherical norm activation function"""
    xnorm = jnp.sqrt(jnp.power(x, 2).sum(axis, keepdims=True) + eps)
    return gamma * x / xnorm + delta


def lagr_spherical_norm(
    x: Array, # input tensor
    gamma: float = 1.0,  # Scale the stdev
    delta: Union[float, jnp.ndarray] = 0.0,  # Shift the mean
    axis: int=-1,  # Which axis to normalize
    eps: float=1e-5,  # Prevent division by 0
) -> Float: # Output scalar
    """Lagrangian of the spherical norm (L2 norm) activation function"""
    y = jnp.sqrt(jnp.power(x, 2).sum(axis, keepdims=True) + eps)
    return (gamma * y + (delta * x).sum()).sum()

# %% ../nbs/01_lagrangians.qmd 29
# Enable this function, but don't document it for now
def lagr_ghostmax(
    a: Array, # Input tensor
    axis: Optional[int] = None, # Axis along which the sum to be computed. If None, the sum is computed along all the axes.
    b: Union[Array, None] = None, # Scaling factors for the exponentials. Must be broadcastable to the shape of a.
    keepdims: bool = False, # If `True`, the axes that are reduced are left in the output as dimensions of size 1.
    return_sign: bool = False, #  If `True`, the output will be a (result, sign) pair, where sign is the sign of the sums and result contains the logarithms of their absolute values. If False only result is returned and it will contain NaN values if the sums are negative.
    ) -> Union[Array, Tuple[Array, Array]]: # Either an array result or a pair of arrays (result, sign), depending on the value of the return_sign argument.
    """ A strictly convex version of logsumexp that concatenates 0 to the array before passing to logsumexp. Code adapted from `jax.nn.logsumexp` (documentation below)
    
    """ + jax.nn.logsumexp.__doc__
    if b is not None:
        a_arr, b_arr = promote_args_inexact("logsumexp", a, b)
        a_arr = jnp.where(b_arr != 0, a_arr, -jnp.inf)
    else:
        a_arr, = promote_args_inexact("logsumexp", a)
        b_arr = a_arr  # for type checking
    pos_dims, dims = _reduction_dims(a_arr, axis)
    amax = jnp.max(a_arr, axis=dims, keepdims=keepdims)
    amax = lax.max(amax, 0.)
    amax = lax.stop_gradient(lax.select(jnp.isfinite(amax), amax, lax.full_like(amax, 0)))
    amax_with_dims = amax if keepdims else lax.expand_dims(amax, pos_dims)

    # fast path if the result cannot be negative.
    if b is None and not np.issubdtype(a_arr.dtype, np.complexfloating):
        out = lax.add(lax.log(jnp.sum(lax.exp(lax.sub(a_arr, amax_with_dims)),
                                      axis=dims, keepdims=keepdims) + lax.exp(-amax)),
                      amax)
        sign = jnp.where(jnp.isnan(out), out, 1.0)
        sign = jnp.where(jnp.isneginf(out), 0.0, sign).astype(out.dtype)
    else:
        expsub = lax.exp(lax.sub(a_arr, amax_with_dims))
        if b is not None:
            expsub = lax.mul(expsub, b_arr)
            
        expsub = expsub + lax.exp(-amax_with_dims)
        sumexp = jnp.sum(expsub, axis=dims, keepdims=keepdims)

        sign = lax.stop_gradient(jnp.sign(sumexp))
        if np.issubdtype(sumexp.dtype, np.complexfloating):
            if return_sign:
                sumexp = sign*sumexp
            out = lax.add(lax.log(sumexp), amax)
        else:
            out = lax.add(lax.log(lax.abs(sumexp)), amax)
    if return_sign:
        return (out, sign)
    if b is not None:
        if not np.issubdtype(out.dtype, np.complexfloating):
            with jax.debug_nans(False):
                out = jnp.where(sign < 0, jnp.array(np.nan, dtype=out.dtype), out)
        return out
    return out

def ghostmax(a, axis=None):
    """Properly implemented ghostmax, robust to input scale. A softmax with additional `1+__` in the denominator. The derivative of `lseplus`"""
    if axis is None:
        og_shape = a.shape
        a = jnp.pad(a.ravel(), ((1,0),), constant_values=0.)
        a = jax.nn.softmax(a)
        return a[1:].reshape(og_shape)
    else:
        pad_widths = [(0,0) for _ in range(len(a.shape))]
        pad_widths[axis] = (1,0)
        pad_width = tuple(pad_widths)
        
        a = jnp.pad(a, pad_width, constant_values=0.)
        a = jax.nn.softmax(a, axis=axis)
        
        out_idxer = [slice(None) for _ in range(len(a.shape))]
        out_idxer[axis] = slice(1, None)
        return a[tuple(out_idxer)]
