"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiObject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const _resolve_1 = require("./_resolve");
const _util_1 = require("./_util");
const chart_1 = require("./chart");
const json_patch_1 = require("./json-patch");
const metadata_1 = require("./metadata");
class ApiObject extends constructs_1.Construct {
    /**
     * Defines an API object.
     *
     * @param scope the construct scope
     * @param id namespace
     * @param props options
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.patches = new Array();
        this.chart = chart_1.Chart.of(this);
        this.kind = props.kind;
        this.apiVersion = props.apiVersion;
        this.apiGroup = parseApiGroup(this.apiVersion);
        this.name = props.metadata?.name ?? this.chart.generateObjectName(this);
        this.metadata = new metadata_1.ApiObjectMetadataDefinition({
            name: this.name,
            // user defined values
            ...props.metadata,
            namespace: props.metadata?.namespace ?? this.chart.namespace,
            labels: {
                ...this.chart.labels,
                ...props.metadata?.labels,
            },
        });
    }
    /**
     * Returns the `ApiObject` named `Resource` which is a child of the given
     * construct. If `c` is an `ApiObject`, it is returned directly. Throws an
     * exception if the construct does not have a child named `Default` _or_ if
     * this child is not an `ApiObject`.
     *
     * @param c The higher-level construct
     */
    static of(c) {
        if (c instanceof ApiObject) {
            return c;
        }
        const child = constructs_1.Node.of(c).defaultChild;
        if (!child) {
            throw new Error(`cannot find a (direct or indirect) child of type ApiObject for construct ${constructs_1.Node.of(c).path}`);
        }
        return ApiObject.of(child);
    }
    /**
     * Create a dependency between this ApiObject and other constructs.
     * These can be other ApiObjects, Charts, or custom.
     *
     * @param dependencies the dependencies to add.
     */
    addDependency(...dependencies) {
        constructs_1.Node.of(this).addDependency(...dependencies);
    }
    /**
     * Applies a set of RFC-6902 JSON-Patch operations to the manifest
     * synthesized for this API object.
     *
     * @param ops The JSON-Patch operations to apply.
     *
     * @example
     *
     *   kubePod.addJsonPatch(JsonPatch.replace('/spec/enableServiceLinks', true));
     *
     */
    addJsonPatch(...ops) {
        this.patches.push(...ops);
    }
    /**
     * Renders the object to Kubernetes JSON.
     *
     * To disable sorting of dictionary keys in output object set the
     * `CDK8S_DISABLE_SORT` environment variable to any non-empty value.
     */
    toJson() {
        const data = {
            ...this.props,
            metadata: this.metadata.toJson(),
        };
        const sortKeys = process.env.CDK8S_DISABLE_SORT ? false : true;
        const json = _util_1.sanitizeValue(_resolve_1.resolve(data), { sortKeys });
        const patched = json_patch_1.JsonPatch.apply(json, ...this.patches);
        // reorder top-level keys so that we first have "apiVersion", "kind" and
        // "metadata" and then all the rest
        const result = {};
        const orderedKeys = ['apiVersion', 'kind', 'metadata', ...Object.keys(patched)];
        for (const k of orderedKeys) {
            if (k in patched) {
                result[k] = patched[k];
            }
        }
        return result;
    }
}
exports.ApiObject = ApiObject;
_a = JSII_RTTI_SYMBOL_1;
ApiObject[_a] = { fqn: "cdk8s.ApiObject", version: "2.3.61" };
function parseApiGroup(apiVersion) {
    const v = apiVersion.split('/');
    // no group means "core"
    // https://kubernetes.io/docs/reference/using-api/api-overview/#api-groups
    if (v.length === 1) {
        return 'core';
    }
    if (v.length === 2) {
        return v[0];
    }
    throw new Error(`invalid apiVersion ${apiVersion}, expecting GROUP/VERSION. See https://kubernetes.io/docs/reference/using-api/api-overview/#api-groups`);
}
//# sourceMappingURL=data:application/json;base64,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