"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = exports.YamlOutputType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const constructs_1 = require("constructs");
const api_object_1 = require("./api-object");
const chart_1 = require("./chart");
const dependency_1 = require("./dependency");
const names_1 = require("./names");
const yaml_1 = require("./yaml");
/** The method to divide YAML output into files */
var YamlOutputType;
(function (YamlOutputType) {
    /** All resources are output into a single YAML file */
    YamlOutputType[YamlOutputType["FILE_PER_APP"] = 0] = "FILE_PER_APP";
    /** Resources are split into seperate files by chart */
    YamlOutputType[YamlOutputType["FILE_PER_CHART"] = 1] = "FILE_PER_CHART";
    /** Each resource is output to its own file */
    YamlOutputType[YamlOutputType["FILE_PER_RESOURCE"] = 2] = "FILE_PER_RESOURCE";
    /** Each chart in its own folder and each resource in its own file */
    YamlOutputType[YamlOutputType["FOLDER_PER_CHART_FILE_PER_RESOURCE"] = 3] = "FOLDER_PER_CHART_FILE_PER_RESOURCE";
})(YamlOutputType = exports.YamlOutputType || (exports.YamlOutputType = {}));
/**
 * Represents a cdk8s application.
 */
class App extends constructs_1.Construct {
    /**
     * Defines an app
     * @param props configuration options
     */
    constructor(props = {}) {
        super(undefined, '');
        this.outdir = props.outdir ?? process.env.CDK8S_OUTDIR ?? 'dist';
        this.outputFileExtension = props.outputFileExtension ?? '.k8s.yaml';
        this.yamlOutputType = props.yamlOutputType ?? YamlOutputType.FILE_PER_CHART;
    }
    /**
     * Synthesize a single chart.
     *
     * Each element returned in the resulting array represents a different ApiObject
     * in the scope of the chart.
     *
     * Note that the returned array order is important. It is determined by the various dependencies between
     * the constructs in the chart, where the first element is the one without dependencies, and so on...
     *
     * @returns An array of JSON objects.
     * @param chart the chart to synthesize.
     * @internal
     */
    static _synthChart(chart) {
        const app = App.of(chart);
        // we must prepare the entire app before synthesizing the chart
        // because the dependency inference happens on the app level.
        resolveDependencies(app);
        // validate the app since we want to call onValidate of the relevant constructs.
        // note this will also call onValidate on constructs from possibly different charts,
        // but thats ok too since we no longer treat constructs as a self-contained synthesis unit.
        validate(app);
        return chartToKube(chart).map(obj => obj.toJson());
    }
    static of(c) {
        const scope = constructs_1.Node.of(c).scope;
        if (!scope) {
            // the app is the only construct without a scope.
            return c;
        }
        return App.of(scope);
    }
    /**
     * Returns all the charts in this app, sorted topologically.
     */
    get charts() {
        const isChart = (x) => x instanceof chart_1.Chart;
        return new dependency_1.DependencyGraph(constructs_1.Node.of(this))
            .topology()
            .filter(isChart);
    }
    /**
     * Synthesizes all manifests to the output directory
     */
    synth() {
        fs.mkdirSync(this.outdir, { recursive: true });
        // Since we plan on removing the distributed synth mechanism, we no longer call `Node.synthesize`, but rather simply implement
        // the necessary operations. We do however want to preserve the distributed validation.
        validate(this);
        // this is kind of sucky, eventually I would like the DependencyGraph
        // to be able to answer this question.
        const hasDependantCharts = resolveDependencies(this);
        const charts = this.charts;
        switch (this.yamlOutputType) {
            case YamlOutputType.FILE_PER_APP:
                let apiObjectList = [];
                for (const chart of charts) {
                    apiObjectList.push(...chartToKube(chart));
                }
                if (charts.length > 0) {
                    yaml_1.Yaml.save(path.join(this.outdir, `app${this.outputFileExtension}`), // There is no "app name", so we just hardcode the file name
                    apiObjectList.map((apiObject) => apiObject.toJson()));
                }
                break;
            case YamlOutputType.FILE_PER_CHART:
                const namer = hasDependantCharts ? new IndexedChartNamer() : new SimpleChartNamer();
                for (const chart of charts) {
                    const chartName = namer.name(chart);
                    const objects = chartToKube(chart);
                    yaml_1.Yaml.save(path.join(this.outdir, chartName + this.outputFileExtension), objects.map(obj => obj.toJson()));
                }
                break;
            case YamlOutputType.FILE_PER_RESOURCE:
                for (const chart of charts) {
                    const apiObjects = chartToKube(chart);
                    apiObjects.forEach((apiObject) => {
                        if (!(apiObject === undefined)) {
                            const fileName = `${`${apiObject.kind}.${apiObject.metadata.name}`
                                .replace(/[^0-9a-zA-Z-_.]/g, '')}`;
                            yaml_1.Yaml.save(path.join(this.outdir, fileName + this.outputFileExtension), [apiObject.toJson()]);
                        }
                    });
                }
                break;
            case YamlOutputType.FOLDER_PER_CHART_FILE_PER_RESOURCE:
                const folderNamer = hasDependantCharts ? new IndexedChartFolderNamer() : new SimpleChartFolderNamer();
                for (const chart of charts) {
                    const chartName = folderNamer.name(chart);
                    const apiObjects = chartToKube(chart);
                    const fullOutDir = path.join(this.outdir, chartName);
                    fs.mkdirSync(fullOutDir, { recursive: true });
                    apiObjects.forEach((apiObject) => {
                        if (!(apiObject === undefined)) {
                            const fileName = `${`${apiObject.kind}.${apiObject.metadata.name}`
                                .replace(/[^0-9a-zA-Z-_.]/g, '')}`;
                            yaml_1.Yaml.save(path.join(fullOutDir, fileName + this.outputFileExtension), [apiObject.toJson()]);
                        }
                    });
                }
                break;
            default:
                break;
        }
    }
    /**
     * Synthesizes the app into a YAML string.
     *
     * @returns A string with all YAML objects across all charts in this app.
     */
    synthYaml() {
        validate(this);
        const charts = this.charts;
        const docs = [];
        for (const chart of charts) {
            const apiObjects = chartToKube(chart);
            docs.push(...apiObjects.map(apiObject => apiObject.toJson()));
        }
        return yaml_1.Yaml.stringify(...docs);
    }
}
exports.App = App;
_a = JSII_RTTI_SYMBOL_1;
App[_a] = { fqn: "cdk8s.App", version: "2.3.61" };
function validate(app) {
    const errors = [];
    for (const child of app.node.findAll()) {
        const childErrors = child.node.validate();
        for (const error of childErrors) {
            errors.push(`[${child.node.path}] ${error}`);
        }
    }
    if (errors.length > 0) {
        throw new Error(`Validation failed with the following errors:\n  ${errors.join('\n  ')}`);
    }
}
function resolveDependencies(app) {
    let hasDependantCharts = false;
    const deps = [];
    for (const child of app.node.findAll()) {
        for (const dep of child.node.dependencies) {
            deps.push({ source: child, target: dep });
        }
    }
    for (const dep of deps) {
        // create explicit api object dependencies from implicit construct dependencies
        const targetApiObjects = constructs_1.Node.of(dep.target).findAll().filter(c => c instanceof api_object_1.ApiObject);
        const sourceApiObjects = constructs_1.Node.of(dep.source).findAll().filter(c => c instanceof api_object_1.ApiObject);
        for (const target of targetApiObjects) {
            for (const source of sourceApiObjects) {
                if (target !== source) {
                    constructs_1.Node.of(source).addDependency(target);
                }
            }
        }
        // create an explicit chart dependency from implicit construct dependencies
        const sourceChart = chart_1.Chart.of(dep.source);
        const targetChart = chart_1.Chart.of(dep.target);
        if (sourceChart !== targetChart) {
            constructs_1.Node.of(sourceChart).addDependency(targetChart);
            hasDependantCharts = true;
        }
    }
    const charts = new dependency_1.DependencyGraph(constructs_1.Node.of(app)).topology()
        .filter(x => x instanceof chart_1.Chart);
    for (const parentChart of charts) {
        for (const childChart of constructs_1.Node.of(parentChart).children.filter(x => x instanceof chart_1.Chart)) {
            // create an explicit chart dependency from nested chart relationships
            constructs_1.Node.of(parentChart).addDependency(childChart);
            hasDependantCharts = true;
        }
    }
    return hasDependantCharts;
}
function chartToKube(chart) {
    return new dependency_1.DependencyGraph(constructs_1.Node.of(chart)).topology()
        .filter(x => x instanceof api_object_1.ApiObject)
        .filter(x => chart_1.Chart.of(x) === chart) // include an object only in its closest parent chart
        .map(x => x);
}
class SimpleChartNamer {
    constructor() {
    }
    name(chart) {
        return `${names_1.Names.toDnsLabel(chart)}`;
    }
}
class IndexedChartNamer extends SimpleChartNamer {
    constructor() {
        super();
        this.index = 0;
    }
    name(chart) {
        const name = `${this.index.toString().padStart(4, '0')}-${super.name(chart)}`;
        this.index++;
        return name;
    }
}
class SimpleChartFolderNamer {
    constructor() {
    }
    name(chart) {
        return names_1.Names.toDnsLabel(chart);
    }
}
class IndexedChartFolderNamer extends SimpleChartFolderNamer {
    constructor() {
        super();
        this.index = 0;
    }
    name(chart) {
        const name = `${this.index.toString().padStart(4, '0')}-${super.name(chart)}`;
        this.index++;
        return name;
    }
}
//# sourceMappingURL=data:application/json;base64,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