"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyVertex = exports.DependencyGraph = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
/**
 * Represents the dependency graph for a given Node.
 *
 * This graph includes the dependency relationships between all nodes in the
 * node (construct) sub-tree who's root is this Node.
 *
 * Note that this means that lonely nodes (no dependencies and no dependants) are also included in this graph as
 * childless children of the root node of the graph.
 *
 * The graph does not include cross-scope dependencies. That is, if a child on the current scope depends on a node
 * from a different scope, that relationship is not represented in this graph.
 *
 */
class DependencyGraph {
    constructor(node) {
        this._fosterParent = new DependencyVertex();
        const nodes = {};
        function putVertex(construct) {
            nodes[constructs_1.Node.of(construct).path] = new DependencyVertex(construct);
        }
        function getVertex(construct) {
            return nodes[constructs_1.Node.of(construct).path];
        }
        // create all vertices of the graph.
        for (const n of node.findAll()) {
            putVertex(n);
        }
        const deps = [];
        for (const child of node.findAll()) {
            for (const dep of child.node.dependencies) {
                deps.push({ source: child, target: dep });
            }
        }
        // create all the edges of the graph.
        for (const dep of deps) {
            if (!getVertex(dep.target)) {
                // dont cross scope boundaries.
                // since charts only renders its own children, this is ok and
                // has the benefit of simplifying the graph. we should reconsider this behavior when moving
                // to a more general purpose use-case.
                continue;
            }
            const sourceDepNode = getVertex(dep.source);
            const targetDepNode = getVertex(dep.target);
            sourceDepNode.addChild(targetDepNode);
        }
        // create the root.
        for (const n of Object.values(nodes)) {
            if (n.inbound.length === 0) {
                // orphans are dependency roots. lets adopt them!
                this._fosterParent.addChild(n);
            }
        }
    }
    /**
     * Returns the root of the graph.
     *
     * Note that this vertex will always have `null` as its `.value` since it is an artifical root
     * that binds all the connected spaces of the graph.
     */
    get root() {
        return this._fosterParent;
    }
    /**
     * @see Vertex.topology()
     */
    topology() {
        return this._fosterParent.topology();
    }
}
exports.DependencyGraph = DependencyGraph;
_a = JSII_RTTI_SYMBOL_1;
DependencyGraph[_a] = { fqn: "cdk8s.DependencyGraph", version: "2.3.61" };
/**
 * Represents a vertex in the graph.
 *
 * The value of each vertex is an `IConstruct` that is accessible via the `.value` getter.
 */
class DependencyVertex {
    constructor(value = undefined) {
        this._children = new Set();
        this._parents = new Set();
        this._value = value;
    }
    /**
     * Returns the IConstruct this graph vertex represents.
     *
     * `null` in case this is the root of the graph.
     */
    get value() {
        return this._value;
    }
    /**
     * Returns the children of the vertex (i.e dependencies)
     */
    get outbound() {
        return Array.from(this._children);
    }
    /**
     * Returns the parents of the vertex (i.e dependants)
     */
    get inbound() {
        return Array.from(this._parents);
    }
    /**
     * Returns a topologically sorted array of the constructs in the sub-graph.
     */
    topology() {
        const found = new Set();
        const topology = [];
        function visit(n) {
            for (const c of n.outbound) {
                visit(c);
            }
            if (!found.has(n)) {
                topology.push(n);
                found.add(n);
            }
        }
        visit(this);
        return topology.filter(d => d.value).map(d => d.value);
    }
    /**
     * Adds a vertex as a dependency of the current node.
     * Also updates the parents of `dep`, so that it contains this node as a parent.
     *
     * This operation will fail in case it creates a cycle in the graph.
     *
     * @param dep The dependency
     */
    addChild(dep) {
        const cycle = dep.findRoute(this);
        if (cycle.length !== 0) {
            cycle.push(dep);
            throw new Error(`Dependency cycle detected: ${cycle.filter(d => d.value).map(d => constructs_1.Node.of(d.value).path).join(' => ')}`);
        }
        this._children.add(dep);
        dep.addParent(this);
    }
    addParent(dep) {
        this._parents.add(dep);
    }
    findRoute(dst) {
        const route = [];
        visit(this);
        return route;
        function visit(n) {
            route.push(n);
            let found = false;
            for (const c of n.outbound) {
                if (c === dst) {
                    route.push(c);
                    return true;
                }
                found = visit(c);
            }
            if (!found) {
                route.pop();
            }
            return found;
        }
    }
}
exports.DependencyVertex = DependencyVertex;
_b = JSII_RTTI_SYMBOL_1;
DependencyVertex[_b] = { fqn: "cdk8s.DependencyVertex", version: "2.3.61" };
//# sourceMappingURL=data:application/json;base64,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