/**
 * Metadata associated with this object.
 */
export interface ApiObjectMetadata {
    /**
     * The unique, namespace-global, name of this object inside the Kubernetes
     * cluster.
     *
     * Normally, you shouldn't specify names for objects and let the CDK generate
     * a name for you that is application-unique. The names CDK generates are
     * composed from the construct path components, separated by dots and a suffix
     * that is based on a hash of the entire path, to ensure uniqueness.
     *
     * You can supply custom name allocation logic by overriding the
     * `chart.generateObjectName` method.
     *
     * If you use an explicit name here, bear in mind that this reduces the
     * composability of your construct because it won't be possible to include
     * more than one instance in any app. Therefore it is highly recommended to
     * leave this unspecified.
     *
     * @default - an app-unique name generated by the chart
     */
    readonly name?: string;
    /**
     * Annotations is an unstructured key value map stored with a resource that may be set by external tools to store and retrieve arbitrary metadata. They are not queryable and should be
     * preserved when modifying objects.
     *
     * @see http://kubernetes.io/docs/user-guide/annotations
     * @default - No annotations.
     */
    readonly annotations?: {
        [key: string]: string;
    };
    /**
     * Map of string keys and values that can be used to organize and categorize (scope and select) objects.
     * May match selectors of replication controllers and services.
     *
     * @see http://kubernetes.io/docs/user-guide/labels
     * @default - No labels.
     */
    readonly labels?: {
        [key: string]: string;
    };
    /**
     * Namespace defines the space within each name must be unique. An empty namespace is equivalent to the "default" namespace, but "default" is the canonical representation.
     * Not all objects are required to be scoped to a namespace - the value of this field for those objects will be empty. Must be a DNS_LABEL. Cannot be updated. More info: http://kubernetes.io/docs/user-guide/namespaces
     *
     * @default undefined (will be assigned to the 'default' namespace)
     */
    readonly namespace?: string;
    /**
     * Namespaced keys that tell Kubernetes to wait until specific conditions are
     * met before it fully deletes resources marked for deletion.
     *
     * Must be empty before the object is deleted from the registry. Each entry is
     * an identifier for the responsible component that will remove the entry from
     * the list. If the deletionTimestamp of the object is non-nil, entries in
     * this list can only be removed. Finalizers may be processed and removed in
     * any order.  Order is NOT enforced because it introduces significant risk of
     * stuck finalizers. finalizers is a shared field, any actor with permission
     * can reorder it. If the finalizer list is processed in order, then this can
     * lead to a situation in which the component responsible for the first
     * finalizer in the list is waiting for a signal (field value, external
     * system, or other) produced by a component responsible for a finalizer later
     * in the list, resulting in a deadlock. Without enforced ordering finalizers
     * are free to order amongst themselves and are not vulnerable to ordering
     * changes in the list.
     *
     * @see https://kubernetes.io/docs/concepts/overview/working-with-objects/finalizers/
     * @default - No finalizers.
     */
    readonly finalizers?: string[];
    /**
     * List of objects depended by this object. If ALL objects in the list have
     * been deleted, this object will be garbage collected. If this object is
     * managed by a controller, then an entry in this list will point to this
     * controller, with the controller field set to true. There cannot be more
     * than one managing controller.
     *
     * Kubernetes sets the value of this field automatically for objects that are
     * dependents of other objects like ReplicaSets, DaemonSets, Deployments, Jobs
     * and CronJobs, and ReplicationControllers. You can also configure these
     * relationships manually by changing the value of this field. However, you
     * usually don't need to and can allow Kubernetes to automatically manage the
     * relationships.
     *
     * @see https://kubernetes.io/docs/concepts/overview/working-with-objects/owners-dependents/
     * @default - automatically set by Kubernetes
     */
    readonly ownerReferences?: OwnerReference[];
    /**
     * Additional metadata attributes.
     */
    readonly [key: string]: any;
}
/**
 * Object metadata.
 */
export declare class ApiObjectMetadataDefinition {
    /**
     * The name of the API object.
     *
     * If a name is specified in `metadata.name` this will be the name returned.
     * Otherwise, a name will be generated by calling
     * `Chart.of(this).generatedObjectName(this)`, which by default uses the
     * construct path to generate a DNS-compatible name for the resource.
     */
    readonly name?: string;
    /**
     * The object's namespace.
     */
    readonly namespace?: string;
    /**
     * Labels associated with this object.
     */
    private readonly labels;
    /**
     * Annotations associated with this object.
     */
    private readonly annotations;
    /**
     * Finalizers associated with this object.
     */
    private readonly finalizers;
    /**
     * Owner references set for this object.
     */
    private readonly ownerReferences;
    /**
     * Additional metadata attributes passed through `options`.
     */
    private readonly _additionalAttributes;
    constructor(options?: ApiObjectMetadata);
    /**
     * Add a label.
     *
     * @param key - The key.
     * @param value - The value.
     */
    addLabel(key: string, value: string): void;
    /**
     * @returns a value of a label or undefined
     * @param key the label
     */
    getLabel(key: string): string | undefined;
    /**
     * Add an annotation.
     *
     * @param key - The key.
     * @param value - The value.
     */
    addAnnotation(key: string, value: string): void;
    /**
     * Add one or more finalizers.
     *
     * @param finalizers the finalizers
     */
    addFinalizers(...finalizers: string[]): void;
    /**
     * Add an owner.
     *
     * @param owner the owner
     */
    addOwnerReference(owner: OwnerReference): void;
    /**
     * Adds an arbitrary key/value to the object metadata.
     * @param key Metadata key
     * @param value Metadata value
     */
    add(key: string, value: any): void;
    /**
     * Synthesizes a k8s ObjectMeta for this metadata set.
     */
    toJson(): any;
}
/**
 * OwnerReference contains enough information to let you identify an owning
 * object. An owning object must be in the same namespace as the dependent, or
 * be cluster-scoped, so there is no namespace field.
 */
export interface OwnerReference {
    /**
     * API version of the referent.
     */
    readonly apiVersion: string;
    /**
     * If true, AND if the owner has the "foregroundDeletion" finalizer, then the
     * owner cannot be deleted from the key-value store until this reference is
     * removed. Defaults to false. To set this field, a user needs "delete"
     * permission of the owner, otherwise 422 (Unprocessable Entity) will be
     * returned.
     *
     * @default false. To set this field, a user needs "delete" permission of the
     * owner, otherwise 422 (Unprocessable Entity) will be returned.
     */
    readonly blockOwnerDeletion?: boolean;
    /**
     * If true, this reference points to the managing controller.
     */
    readonly controller?: boolean;
    /**
     * Kind of the referent.
     *
     * @see https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#types-kinds
     */
    readonly kind: string;
    /**
     * Name of the referent.
     *
     * @see http://kubernetes.io/docs/user-guide/identifiers#names
     */
    readonly name: string;
    /**
     * UID of the referent.
     *
     * @see http://kubernetes.io/docs/user-guide/identifiers#uids
     */
    readonly uid: string;
}
