"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Names = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const constructs_1 = require("constructs");
const MAX_LEN = 63;
const VALIDATE = /^[0-9a-z-]+$/;
const VALIDATE_LABEL_VALUE = /^(([0-9a-zA-Z][0-9a-zA-Z-_.]*)?[0-9a-zA-Z])?$/;
const HASH_LEN = 8;
/**
 * Utilities for generating unique and stable names.
 */
class Names {
    /* istanbul ignore next */
    constructor() {
        return;
    }
    /**
     * Generates a unique and stable name compatible DNS_LABEL from RFC-1123 from
     * a path.
     *
     * The generated name will:
     *  - contain at most 63 characters
     *  - contain only lowercase alphanumeric characters or ‘-’
     *  - start with an alphanumeric character
     *  - end with an alphanumeric character
     *
     * The generated name will have the form:
     *  <comp0>-<comp1>-..-<compN>-<short-hash>
     *
     * Where <comp> are the path components (assuming they are is separated by
     * "/").
     *
     * Note that if the total length is longer than 63 characters, we will trim
     * the first components since the last components usually encode more meaning.
     *
     * @link https://tools.ietf.org/html/rfc1123
     *
     * @param scope The construct for which to render the DNS label
     * @param options Name options
     * @throws if any of the components do not adhere to naming constraints or
     * length.
     */
    static toDnsLabel(scope, options = {}) {
        const maxLen = options.maxLen ?? MAX_LEN;
        const delim = options.delimiter ?? '-';
        const include_hash = options.includeHash ?? true;
        if (maxLen < HASH_LEN && include_hash) {
            throw new Error(`minimum max length for object names is ${HASH_LEN} (required for hash)`);
        }
        const node = constructs_1.Node.of(scope);
        let components = node.path.split('/');
        components.push(...options.extra ?? []);
        // special case: if we only have one component in our path and it adheres to DNS_NAME, we don't decorate it
        if (components.length === 1 && VALIDATE.test(components[0]) && components[0].length <= maxLen) {
            return components[0];
        }
        // okay, now we need to normalize all components to adhere to DNS_NAME and append the hash of the full path.
        components = components.map(c => normalizeToDnsName(c, maxLen));
        if (include_hash) {
            components.push(calcHash(node, HASH_LEN));
        }
        return toHumanForm(components, delim, maxLen);
    }
    /**
     * Generates a unique and stable name compatible label key name segment and
     * label value from a path.
     *
     * The name segment is required and must be 63 characters or less, beginning
     * and ending with an alphanumeric character ([a-z0-9A-Z]) with dashes (-),
     * underscores (_), dots (.), and alphanumerics between.
     *
     * Valid label values must be 63 characters or less and must be empty or
     * begin and end with an alphanumeric character ([a-z0-9A-Z]) with dashes
     * (-), underscores (_), dots (.), and alphanumerics between.
     *
     * The generated name will have the form:
     *  <comp0><delim><comp1><delim>..<delim><compN><delim><short-hash>
     *
     * Where <comp> are the path components (assuming they are is separated by
     * "/").
     *
     * Note that if the total length is longer than 63 characters, we will trim
     * the first components since the last components usually encode more meaning.
     *
     * @link https://kubernetes.io/docs/concepts/overview/working-with-objects/labels/#syntax-and-character-set
     *
     * @param scope The construct for which to render the DNS label
     * @param options Name options
     * @throws if any of the components do not adhere to naming constraints or
     * length.
     */
    static toLabelValue(scope, options = {}) {
        const maxLen = options.maxLen ?? MAX_LEN;
        const delim = options.delimiter ?? '-';
        const include_hash = options.includeHash ?? true;
        if (maxLen < HASH_LEN && include_hash) {
            throw new Error(`minimum max length for label is ${HASH_LEN} (required for hash)`);
        }
        if (/[^0-9a-zA-Z-_.]/.test(delim)) {
            throw new Error('delim should not contain "[^0-9a-zA-Z-_.]"');
        }
        const node = constructs_1.Node.of(scope);
        let components = node.path.split('/');
        components.push(...options.extra ?? []);
        // special case: if we only have one component in our path and it adheres to DNS_NAME, we don't decorate it
        if (components.length === 1 && VALIDATE_LABEL_VALUE.test(components[0]) && components[0].length <= maxLen) {
            return components[0];
        }
        // okay, now we need to normalize all components to adhere to label and append the hash of the full path.
        components = components.map(c => normalizeToLabelValue(c, maxLen));
        if (include_hash) {
            components.push(calcHash(node, HASH_LEN));
        }
        const result = toHumanForm(components, delim, maxLen);
        // slicing might let '-', '_', '.' be in the start of the result.
        return result.replace(/^[^0-9a-zA-Z]+/, '');
    }
}
exports.Names = Names;
_a = JSII_RTTI_SYMBOL_1;
Names[_a] = { fqn: "cdk8s.Names", version: "2.3.61" };
function omitDuplicates(value, index, components) {
    return value !== components[index - 1];
}
function omitDefaultChild(value, _, __) {
    return value.toLowerCase() !== 'resource' && value.toLowerCase() !== 'default';
}
function toHumanForm(components, delim, maxLen) {
    return components.reverse()
        .filter(omitDuplicates)
        .join('/')
        .slice(0, maxLen)
        .split('/')
        .reverse()
        .filter(x => x)
        .join(delim)
        .split(delim)
        .filter(x => x)
        .filter(omitDefaultChild)
        .join(delim);
}
function normalizeToDnsName(c, maxLen) {
    return c
        .toLocaleLowerCase() // lower case
        .replace(/[^0-9a-zA-Z-_.]/g, '') // remove non-allowed characters
        .substr(0, maxLen); // trim to maxLength
}
function calcHash(node, maxLen) {
    if (process.env.CDK8S_LEGACY_HASH) {
        const hash = crypto.createHash('sha256');
        hash.update(node.path);
        return hash.digest('hex').slice(0, maxLen);
    }
    return node.addr.substring(0, HASH_LEN);
}
function normalizeToLabelValue(c, maxLen) {
    return c
        .replace(/[^0-9a-zA-Z-_.]/g, '') // remove non-allowed characters
        .substr(0, maxLen); // trim to maxLength
}
//# sourceMappingURL=data:application/json;base64,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