"""Core task implementation."""

from __future__ import annotations

import abc
from collections.abc import Iterable
from typing import TYPE_CHECKING, Any

from sieves.data import Doc
from sieves.serialization import Attribute, Config

if TYPE_CHECKING:
    # Imported only for type checking to avoid import cycles at runtime.
    from sieves.pipeline import Pipeline


class Task(abc.ABC):
    """Abstract base class for tasks that can be executed on documents."""

    def __init__(self, task_id: str | None, include_meta: bool, batch_size: int):
        """
        Initiate new Task.

        :param task_id: Task ID.
        :param include_meta: Whether to include meta information generated by the task.
        :param batch_size: Batch size for processing documents. Use -1 to process all documents at once.
        """
        self._task_id = task_id if task_id else self.__class__.__name__
        self._include_meta = include_meta
        self._batch_size = batch_size

    @property
    def id(self) -> str:
        """Return task ID.

        Used by pipeline for results and dependency management.

        :return: Task ID.
        """
        return self._task_id

    @abc.abstractmethod
    def __call__(self, docs: Iterable[Doc]) -> Iterable[Doc]:
        """Execute task.

        :param docs: Docs to process.
        :return: Processed docs.
        """

    def __add__(self, other: Task | Pipeline) -> Pipeline:
        """Chain this task with another task or pipeline using the ``+`` operator.

        This returns a new ``Pipeline`` that executes this task first, followed by the
        task(s) in ``other``. The original task(s)/pipeline are not mutated.

        Cache semantics:
        - If ``other`` is a ``Pipeline``, the resulting pipeline adopts ``other``'s
          ``use_cache`` setting (because the left-hand side is a single task).
        - If ``other`` is a ``Task``, the resulting pipeline defaults to ``use_cache=True``.

        :param other: A ``Task`` or ``Pipeline`` to execute after this task.
        :return: A new ``Pipeline`` representing the chained execution.
        :raises TypeError: If ``other`` is not a ``Task`` or ``Pipeline``.
        """
        # Lazy import to avoid circular dependency at module import time.
        from sieves.pipeline import Pipeline

        if isinstance(other, Pipeline):
            return Pipeline(tasks=[self, *other.tasks], use_cache=other.use_cache)

        if isinstance(other, Task):
            return Pipeline(tasks=[self, other])

        raise TypeError(f"Cannot chain Task with {type(other).__name__}")

    @property
    def _state(self) -> dict[str, Any]:
        """Return attributes to serialize.

        :return: Dict of attributes to serialize.
        """
        return {
            "task_id": self._task_id,
            "include_meta": self._include_meta,
            "batch_size": self._batch_size,
        }

    def serialize(self) -> Config:
        """Serialize task.

        :return: Config instance.
        """
        return Config.create(self.__class__, {k: Attribute(value=v) for k, v in self._state.items()})

    @classmethod
    def deserialize(cls, config: Config, **kwargs: dict[str, Any]) -> Task:
        """Generate Task instance from config.

        :param config: Config to generate instance from.
        :param kwargs: Values to inject into loaded config.
        :return: Deserialized Task instance.
        """
        # Deserialize and inject engine.
        return cls(**config.to_init_dict(cls, **kwargs))
