import warnings
from collections.abc import Mapping
from typing import Any

import numpy as np
import scipy.sparse as sp
from anndata import AnnData

from exprmat.utils import choose_representation as choose_rep
from exprmat.ansi import warning, info, error


def cnv_score(
    adata: AnnData,
    groupby: str = "leiden.cnv",
    *,
    use_rep: str = "cnv",
    key_added: str = "score.cnv"
) -> Mapping[Any, np.number] | None:
    """
    Assign each CNV cluster a CNV score.

    Clusters with a high score are likely affected by copy number abberations.
    Based on this score, cells can be divided into tumor / normal cells. Ths score 
    is currently simply defined as the mean of the absolute values of the
    CNV scores in each cluster.

    Parameters
    ----------
    adata
        Annotated data matrix

    groupby
        Key under which the clustering is stored in adata.obs. Usually the result of 
        CNV leiden clustering, but could also be other grouping information, e.g. 
        sample or patient information.

    use_rep
        Key under which the result of ``infercnv`` stored in adata.

    key_added
        Key under which the score will be stored in `adata.obs`.
    """

    if groupby not in adata.obs.columns:
        error(f"`{groupby}` not found in `adata.obs`.")

    cluster_score = {
        cluster: np.mean(np.abs(adata.obsm[f"{use_rep}"][adata.obs[groupby].values == cluster, :]))
        for cluster in adata.obs[groupby].unique()
    }

    score_array = np.array([cluster_score[c] for c in adata.obs[groupby]])
    adata.obs[key_added] = score_array


def ithgex(
    adata: AnnData,
    groupby: str,
    *,
    use_raw: bool | None = None,
    layer: str | None = None,
    key_added: str = "score.ithgex",
) -> Mapping[str, float] | None:
    """
    Compute the ITHGEX diversity score based on gene expression. A high score indicates a 
    high diversity of gene expression profiles of cells within a group.

    The score is defined as follows:

    Intratumoral heterogeneity scores based on CNAs and gene expressions
    The calculations of intratumoral heterogeneity scores were inspired by a
    previous study and modified as follows. First, to calculate ITHCNA, we used
    the relative expression value matrix generated by inferCNV and calculated the
    pairwise cell-cell distances using Pearson's correlation coefficients for each
    patient. ITHCNA was defined as interquartile range (IQR) of the distribution for
    all malignant cell pairs' Pearson's correlation coefficients. **Similarly, we also
    used gene expression profiles of cancer cells of each patient to construct the
    distribution of the intratumoral distances. ITHGEX was assigned as the IQR of the
    distribution.**

    Parameters
    ----------
    adata
        Annotated data matrix

    groupby
        Calculate diversity for each group defined in this category.
    """
    
    groups = adata.obs[groupby].unique()
    ithgex = {}
    
    for group in groups:
        tmp_adata = adata[adata.obs[groupby] == group, :]
        X = choose_rep(tmp_adata, use_raw, layer)
        if sp.issparse(X): X = X.todense()
        if X.shape[0] <= 1: continue
        
        pcorr = np.corrcoef(X, rowvar = True)
        assert pcorr.shape == (
            tmp_adata.shape[0],
            tmp_adata.shape[0],
        ), f"pcorr is a cell by cell matrix {tmp_adata.shape[0]} {pcorr.shape}"
        
        q75, q25 = np.percentile(pcorr, [75, 25])
        ithgex[group] = q75 - q25

    ithgex_obs = np.empty(adata.shape[0])
    for group in groups:
        ithgex_obs[adata.obs[groupby] == group] = ithgex[group]
    adata.obs[key_added] = ithgex_obs


def ithcna(
    adata: AnnData,
    groupby: str,
    *,
    use_rep: str = "cnv",
    key_added: str = "score.ithcna"
) -> Mapping[str, float] | None:
    """
    Compute the ITHCNA diversity score based on copy number variation.
    A high score indicates a high diversity of CNV profiles of cells within a group.

    The score is defined as follows:

    Intratumoral heterogeneity scores based on CNAs and gene expressions
    The calculations of intratumoral heterogeneity scores were inspired by a
    previous study and modified as follows. First, to calculate ITHCNA, we used
    the relative expression value matrix generated by inferCNV and calculated the
    pairwise cell-cell distances using Pearson's correlation coefficients for each
    patient. ITHCNA was defined as interquartile range (IQR) of the distribution for
    all malignant cell pairs' Pearson's correlation coefficients.

    Parameters
    ----------
    adata
        Annotated data matrix

    groupby
        Calculate diversity for each group defined in this category.
    """

    groups = adata.obs[groupby].unique()
    ithcna = {}

    for group in groups:
        tmp_adata = adata[adata.obs[groupby] == group, :]
        X = tmp_adata.obsm[use_rep]
        if sp.issparse(X): X = X.todense()
        if X.shape[0] <= 1: continue

        pcorr = np.corrcoef(X, rowvar=True)
        assert pcorr.shape == (
            tmp_adata.shape[0],
            tmp_adata.shape[0],
        ), "pcorr is a cell by cell matrix"
        
        q75, q25 = np.percentile(pcorr, [75, 25])
        ithcna[group] = q75 - q25

    
    ithcna_obs = np.empty(adata.shape[0])
    for group in groups:
        ithcna_obs[adata.obs[groupby] == group] = ithcna[group]
    adata.obs[key_added] = ithcna_obs