'''
Summarize dataset. The summarizing functions summarizes a cell by gene matrix anndata into
category by summary matrix. You should specify a certain categorical metadata to summarize with,
and the target variables to merge. For example, you may calculate the mean expression of
certain gene across category of cell types.

The resulting dataframe is a variant of another smaller annotated data. it is at most three
dimensional with the third dimension extends into categorical layers. The resulting matrix
is of n(on) rows and n(across) columns, with n(split) layers. The on argument is mandatory,
whereas across and split can be None. If it is none, the autogenerated 'all' category will
be appended and adopted.
'''

import numpy as np
import anndata as ad
import pandas as pd
import scipy.sparse as sp

from exprmat.ansi import error, info, warning
from exprmat.utils import choose_layer


def summarize(
    adata, data = 'X', method = 'n', method_args = {},
    
    orient = 'obs', 
    on = 'sample', 
    across = None,
    split = None, 

    attached_metadata_on = None, attached_metadata_across = None,
    attach_method_on = 'first', attach_method_across = 'first'
):

    # the orient switch determines the variables to choose from
    var_df = None
    if orient == 'obs': var_df = adata.obs
    elif orient == 'var': var_df = adata.var
    else: error('orient can only be one of `obs` and `var`.')

    # build adequate subset on `on`, `across` and `split`.
    on_vals = var_df[on].value_counts().index.tolist()
    across_vals = var_df[across].value_counts().index.tolist() if across is not None else ['.all']
    split_vals = var_df[split].value_counts().index.tolist() if split is not None else ['.all']

    on_col = var_df[on].tolist()
    across_col = var_df[across].tolist() if across is not None else ['.all'] * len(var_df)
    split_col = var_df[split].tolist() if split is not None else ['.all'] * len(var_df)

    summary_layers = {}
    count = 0; total = len(split_vals) * len(across_vals) * len(on_vals)
    for x_split in split_vals:
        mat = np.zeros((len(on_vals), len(across_vals))).tolist()
        for i_across in range(len(across_vals)):
            for i_on in range(len(on_vals)):

                mask = var_df[on] == on_vals[i_on]
                if split is not None: mask &= var_df[split] == x_split
                if mask.sum() == 0: 
                    mat[i_on][i_across] = float('nan')
                    continue

                if across is not None: mask &= var_df[across] == across_vals[i_across]
                
                count += 1
                val = float('nan')
                nelems = mask.sum()

                # apply the method to adata sections.
                # however, for consideration of speed, some magic method do not need the real
                # subsetting of the annotated data.

                if isinstance(method, str):
                    if method == 'n':
                        val = nelems
                    else: error(f'{method}: unrecognized magic shortcut method.')

                elif nelems > 0:
                    s = choose_layer(adata, layer = data)
                    s = s[mask, :].copy() if orient == 'obs' else s[:, mask].copy()
                    val = method(s, **method_args)

                mat[i_on][i_across] = val
                pass
            pass

        summary_layers[x_split] = np.array(mat)
        pass

    summ = ad.AnnData(
        X = sp.csr_matrix((len(on_vals), len(across_vals)), dtype = np.float32)
    )

    summ.obs_names = on_vals
    summ.var_names = across_vals
    for k in summary_layers.keys(): summ.layers[k] = summary_layers[k]

    # here, we try to merge specified metadata columns onto the summary adata.
    # for categorical data, the methods 'first' and 'most' are available.
    # for numerical data, methods may be 'mean', 'median', etc.

    if attached_metadata_on is not None:
        if isinstance(attach_method_on, str): attach_method_on = [attach_method_on] * len(on_vals)
        for meta, meth in zip(attached_metadata_on, attach_method_on):
            meta_col = var_df[meta]
            result = []
            for i_on in range(len(on_vals)):
                subset = [x == on_vals[i_on] for x in on_col]
                subset = meta_col.loc[subset].value_counts().index.tolist()
                result.append(None if len(subset) == 0 else subset[0])

            summ.obs[meta] = result

    if attached_metadata_across is not None:
        if isinstance(attach_method_across, str): attach_method_across = \
            [attach_method_across] * len(across_vals)
        for meta, meth in zip(attached_metadata_across, attach_method_across):
            meta_col = var_df[meta]
            result = []
            for i_across in range(len(across_vals)):
                subset = [x == across_vals[i_across] for x in across_vals]
                subset = meta_col.loc[subset].value_counts().index.tolist()
                result.append(None if len(subset) == 0 else subset[0])
            
            summ.var[meta] = result

    return summ


def normalize_row(adata, layer = 'X'):
    
    mat = choose_layer(adata, layer = layer)
    rowsum = np.nansum(mat, axis = 1) + 1e-5
    mat = (mat.T / rowsum).T

    if layer == 'X': adata.X = mat
    else: adata.layers[layer] = mat


def normalize_column(adata, layer = 'X'):
    
    mat = choose_layer(adata, layer = layer)
    rowsum = np.nansum(mat, axis = 0) + 1e-5
    mat = mat / rowsum

    if layer == 'X': adata.X = mat
    else: adata.layers[layer] = mat


def to_long_form(
    adata, *, 
    selected_rows = None, 
    selected_cols = None, 
    selected_layers = None
):
    brow = [x in selected_rows for x in adata.obs_names.tolist()] \
        if selected_rows is not None else [True] * adata.n_obs
    bcol = [x in selected_cols for x in adata.var_names.tolist()] \
        if selected_cols is not None else [True] * adata.n_vars
    
    adata = adata[brow, bcol].copy()
    if selected_layers is not None:
        for x in list(adata.layers.keys()):
            if x not in selected_layers: del adata.layers[x]

    # construct the full long form data table with three dimensions
    
    coldim = adata.var_names.tolist()
    rowdim = adata.obs_names.tolist()
    zdim = list(adata.layers.keys())

    data = {
        'value': [],
        'row': [],
        'column': [],
        'layer': []
    }

    for coldata in adata.var.columns.tolist():
        data[coldata] = []
    for rowdata in adata.obs.columns.tolist():
        data[rowdata] = []

    for z in zdim:
        l = choose_layer(adata, layer = z)
        for ix in range(len(rowdim)):
            for iy in range(len(coldim)):
                data['value'].append(l[ix, iy])
                data['column'].append(coldim[iy])
                for coldata in adata.var.columns.tolist():
                    data[coldata].append(adata.var.loc[coldim[iy], coldata])

            data['row'] += [rowdim[ix]] * len(coldim)
            for rowdata in adata.obs.columns.tolist():
                data[rowdata] += [adata.obs.loc[rowdim[ix], rowdata]] * len(coldim)
        data['layer'] += [z] * (len(rowdim) * len(coldim))
    
    return pd.DataFrame(data)
