# Generated by Django 4.2.11 on 2024-06-10 12:21

from django.db import migrations
from django.db.migrations import RunPython
from edc_utils import get_utcnow
from tqdm import tqdm

from effect_screening.eligibility import ScreeningEligibility


class DataMigrationError(Exception):
    pass


def re_assess_serum_crag_date_elig(apps, schema_editor):
    """Re-assess eligibility for Subject Screening instances where
    reasons_ineligible currently contains "Serum CrAg > 14 days".

    Eligibility criteria has been relaxed to allow CrAg date up to
    21 days (#488-3).
    """
    model_cls = apps.get_model("effect_screening.subjectscreening")
    qs = model_cls.objects.all()
    total = qs.count()

    re_assessed_count = re_assessed_gt_21_count = 0

    print(f"\nProcessing {total} Subject Screening instances ...")
    for obj in tqdm(qs, total=total):
        days_between = (obj.report_datetime.date() - obj.serum_crag_date).days

        if obj.reasons_ineligible and "Serum CrAg > 14 days" in obj.reasons_ineligible:
            print(
                " * Re-assessing eligibility for "
                f"'{obj.screening_identifier}' {days_between=} ..."
            )
            if not days_between > 14:
                raise DataMigrationError(f"Expected days_between > 14. Got {days_between=}.")

            reasons_before = obj.reasons_ineligible
            eligibility = ScreeningEligibility(obj)
            eligibility.assess_eligibility()

            obj.modified = get_utcnow()
            obj.user_modified = __name__ if len(__name__) <= 50 else f"{__name__[:46]} ..."
            obj.save_base(
                update_fields=[
                    "reasons_ineligible",
                    "modified",
                    "user_modified",
                ]
            )

            if obj.reasons_ineligible and obj.reasons_ineligible == reasons_before:
                raise DataMigrationError(
                    "Expected reasons_ineligible to be updated. "
                    f"Got '{obj.screening_identifier}' "
                    f"{reasons_before=} {obj.reasons_ineligible=}."
                )
            re_assessed_count += 1
            if days_between > 21:
                re_assessed_gt_21_count += 1

        # Validate assumptions about Serum CrAg date eligibility
        # in ALL screening instances
        if days_between > 21 and (
            not obj.reasons_ineligible or "Serum CrAg > 21 days" not in obj.reasons_ineligible
        ):
            raise DataMigrationError(
                "Expected 'Serum CrAg > 21 days' in reasons_ineligible. "
                f"Got '{obj.screening_identifier}' {obj.reasons_ineligible=}."
            )
        elif (
            days_between <= 21
            and obj.reasons_ineligible
            and "Serum CrAg >" in obj.reasons_ineligible
        ):
            raise DataMigrationError(
                "Did not expect 'Serum CrAg > ' in reasons_ineligible. "
                f"Got '{obj.screening_identifier}' {obj.reasons_ineligible=}."
            )
        elif obj.reasons_ineligible and "Serum CrAg > 14 days" in obj.reasons_ineligible:
            raise DataMigrationError(
                "Expected all instances of 'Serum CrAg > 14 days' to "
                "be either removed or replaced with 'Serum CrAg > 21 days' "
                f"Got '{obj.screening_identifier}' {obj.reasons_ineligible}."
            )

    print(f"Processed {total=} Subject Screening instances.")
    print(f"{re_assessed_count=}")
    print(f"{re_assessed_gt_21_count=}")
    print("Done.")


class Migration(migrations.Migration):

    dependencies = [
        ("effect_screening", "0048_alter_historicalsubjectscreening_report_datetime_and_more"),
    ]

    operations = [RunPython(re_assess_serum_crag_date_elig)]
