# Nado Protocol Python SDK

This is the Python SDK for the [Nado Protocol API](TODO).

See [SDK docs](https://nadohq.github.io/nado-python-sdk/index.html) to get started.

## Requirements

- Python 3.9 or above

## Installation

You can install the SDK via pip:

```bash
pip install nado-protocol
```

## Basic usage

### Import the necessary utilities:

```python
from nado_protocol.client import create_nado_client, NadoClientMode
from nado_protocol.contracts.types import DepositCollateralParams
from nado_protocol.engine_client.types.execute import (
    OrderParams,
    PlaceOrderParams,
    SubaccountParams
)
from nado_protocol.utils.expiration import OrderType, get_expiration_timestamp
from nado_protocol.utils.math import to_pow_10, to_x18
from nado_protocol.utils.nonce import gen_order_nonce
from nado_protocol.utils.order import build_appendix
```

### Create the NadoClient providing your private key:

```python
print("setting up nado client...")
private_key = "xxx"
client = create_nado_client(NadoClientMode.DEVNET, private_key)
```

### Perform basic operations:

```python
# Depositing collaterals
print("approving allowance...")
approve_allowance_tx_hash = client.spot.approve_allowance(0, to_pow_10(100000, 6))
print("approve allowance tx hash:", approve_allowance_tx_hash)

print("querying my allowance...")
token_allowance = client.spot.get_token_allowance(0, client.context.signer.address)
print("token allowance:", token_allowance)

print("depositing collateral...")
deposit_tx_hash = client.spot.deposit(
   DepositCollateralParams(
      subaccount_name="default", product_id=0, amount=to_pow_10(100000, 6)
   )
)
print("deposit collateral tx hash:", deposit_tx_hash)

# Placing orders
print("placing order...")
owner = client.context.engine_client.signer.address
product_id = 1
order = OrderParams(
   sender=SubaccountParams(
      subaccount_owner=owner,
      subaccount_name="default",
   ),
   priceX18=to_x18(20000),
   amount=to_pow_10(1, 17),
   expiration=get_expiration_timestamp(40),
   nonce=gen_order_nonce(),
   appendix=build_appendix(order_type=OrderType.POST_ONLY)
)
res = client.market.place_order({"product_id": product_id, "order": order})
print("order result:", res.json(indent=2))
```

See [Getting Started](https://nadohq.github.io/nado-python-sdk/getting-started.html) for more.

## Running locally

1. Clone [github repo](https://github.com/nadohq/nado-python-sdk)

2. Install poetry

```

$ curl -sSL https://install.python-poetry.org | python3 -

```

3. Setup a virtual environment and activate it

```

$ python3 -m venv venv
$ source ./venv/bin/activate

```

4. Install dependencies via `poetry install`
5. Setup an `.env` file and set the following envvars

```shell
CLIENT_MODE='devnet'
SIGNER_PRIVATE_KEY="0x..."
LINKED_SIGNER_PRIVATE_KEY="0x..." # not required
```

### Run tests

```
$ poetry run test
```

### Run sanity checks

- `poetry run client-sanity`: runs sanity checks for the top-level client.
- `poetry run engine-sanity`: runs sanity checks for the `engine-client`.
- `poetry run indexer-sanity`: runs sanity checks for the `indexer-client`.
- `poetry run contracts-sanity`: runs sanity checks for the contracts module.

### Build Docs

To build the docs locally run:

```
$ poetry run sphinx-build docs/source docs/build
```
