"""JUNO.md generator utilities (UI-agnostic).

Creates a comprehensive JUNO.md with stable sections used by verification.
"""

from __future__ import annotations

import platform
from datetime import datetime
from dataclasses import dataclass
from pathlib import Path
from typing import Any, Dict, Optional


@dataclass
class JunoMdInputs:
    workdir: Path
    project_description: str = ""
    selected_editor: str = "Unknown"
    ai_analysis: str = ""
    detected_deps: Dict[str, Any] = None
    fetched_docs: Dict[str, Any] = None


class JunoMdGenerator:
    def __init__(self, logger: Optional[Any] = None) -> None:
        self.log = logger

    def generate_content(self, inputs: JunoMdInputs) -> str:
        project_name = Path(inputs.workdir).name
        deps = inputs.detected_deps or {}
        docs = inputs.fetched_docs or {}

        if self.log:
            self.log.debug(
                "juno_md_generate_content",
                workdir=str(inputs.workdir),
                deps=len(deps.get("dependencies", [])),
            )

        lines = []
        lines.append(f"# JUNO Development Guide - {project_name}")
        lines.append("")
        lines.append("## Project Analysis")
        lines.append("This guide helps AI assistants reason about your project.")
        lines.append("")
        lines.append("### Basic Information")
        lines.append(f"- **Project Path**: `{inputs.workdir}`")
        lines.append(f"- **Project Type**: {deps.get('project_type', 'Unknown')}")
        lines.append(f"- **Primary Language**: {deps.get('language', 'Unknown')}")
        lines.append(f"- **Setup Date**: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
        lines.append(f"- **Platform**: {platform.system()} ({platform.machine()})")
        lines.append(f"- **Selected AI IDE**: {inputs.selected_editor}")
        lines.append("")
        lines.append("### Project Description")
        lines.append(inputs.project_description or "No description provided")
        lines.append("")
        lines.append("## Architecture & Dependencies")
        lines.append("")
        lines.append("### Detected Dependencies")
        dep_list = deps.get("dependencies", [])
        if dep_list:
            for dep in dep_list[:15]:
                lines.append(f"- `{dep}`")
            if len(dep_list) > 15:
                lines.append(f"- ... and {len(dep_list) - 15} more dependencies")
        else:
            lines.append("- No dependencies detected")
        lines.append("")
        lines.append("### Package Files")
        for p in deps.get("package_files", []) or []:
            lines.append(f"- `{p}`")
        if not deps.get("package_files"):
            lines.append("- No package files detected")
        lines.append("")
        lines.append("## AI Analysis Results")
        if inputs.ai_analysis and len(inputs.ai_analysis) > 0:
            summary = inputs.ai_analysis[:1000] + ("..." if len(inputs.ai_analysis) > 1000 else "")
            lines.append("```")
            lines.append(summary)
            lines.append("```")
        else:
            lines.append("AI analysis not available in headless mode.")
        lines.append("")
        lines.append("## External Context")
        lines.append("The `external_context/` directory contains dependency docs:")
        saved_files = docs.get("saved_files", [])
        if saved_files:
            for file_info in saved_files:
                dep_name = file_info.get('dependency', 'unknown')
                filename = file_info.get('filename', f"{dep_name}.md")
                lines.append(f"- **{dep_name}**: `external_context/{filename}`")
        else:
            lines.append("- No external documentation was fetched during setup")
        lines.append("")
        lines.append("## MCP Server")
        lines.append("If configured, VibeContext MCP provides tools to resolve libraries and fetch documentation.")
        lines.append("")
        lines.append("## Development Workflows")
        lines.append("1. Reference external_context/ for dependency docs")
        lines.append("2. Use AI IDE intelligently with project context")
        lines.append("3. Write tests first for critical paths")
        lines.append("")
        lines.append("## Best Practices")
        lines.append(f"- Follow {deps.get('language', 'language')}-specific conventions")
        lines.append("- Keep documentation in sync with code")
        lines.append("- Validate AI-generated code against requirements")
        lines.append("")
        lines.append("---")
        lines.append(
            f"*Generated by juno-agent on {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}*"
        )
        return "\n".join(lines) + "\n"

    def write(self, inputs: JunoMdInputs) -> Path:
        content = self.generate_content(inputs)
        path = Path(inputs.workdir) / "JUNO.md"
        path.write_text(content, encoding="utf-8")
        if self.log:
            self.log.info("juno_md_written", path=str(path))
        return path


def generate_and_write_juno_md(
    workdir: Path,
    project_description: str,
    selected_editor: str,
    ai_analysis: str,
    detected_deps: Dict[str, Any],
    fetched_docs: Dict[str, Any],
    logger: Optional[Any] = None,
) -> Path:
    gen = JunoMdGenerator(logger=logger)
    return gen.write(
        JunoMdInputs(
            workdir=Path(workdir),
            project_description=project_description,
            selected_editor=selected_editor,
            ai_analysis=ai_analysis,
            detected_deps=detected_deps or {},
            fetched_docs=fetched_docs or {},
        )
    )
