"""Shared Setup Pipeline used by headless and TUI.

This module orchestrates the full setup flow:
- Scan project
- Initialize external_context and symlink
- Fetch docs via AgenticDependencyResolver (with optional Textual UI callback)
- Generate JUNO.md and AGENTS.md/CLAUDE.md (prepend agentic contributor guide)
- Install MCP (Claude via CLI, Cursor/others via config file)
- Persist editor + mcp_server_installed flags
- Verify using VerifyAgent; optionally write a report file
"""

from __future__ import annotations

from pathlib import Path
from typing import Any, Dict, Optional


def run_setup_pipeline(
    workdir: Path,
    config_manager,
    editor_display: str,
    logger: Optional[Any] = None,
    report_file: Optional[Path] = None,
    textual_ui_callback: Optional[Any] = None,
    project_description: Optional[str] = None,
) -> Dict[str, Any]:
    from ..fancy_ui.setup.dependency_scanner import DependencyScanner
    from ..fancy_ui.setup.external_context_manager import ExternalContextManager
    from ..fancy_ui.setup.agentic_dependency_resolver import AgenticDependencyResolver
    from ..setup import (
        generate_and_write_agents_md,
        generate_and_write_juno_md,
        AgentsMdGenerator,
        AgentsMdInputs,
        VerifyAgent,
    )
    from ..fancy_ui.setup import MCPInstaller, MCPInstallationError

    wd = Path(workdir)
    dbg = logger or (config_manager.create_debug_logger(debug=True) if hasattr(config_manager, 'create_debug_logger') else None)

    # 1) Scan project
    scanner = DependencyScanner(project_path=wd)
    scan = scanner.scan_project_dependencies()
    detected = {
        "project_type": scan.get("metadata", {}).get("project_type", "Unknown"),
        "language": scan.get("language", "Unknown"),
        "dependencies": scan.get("dependencies", []),
        "package_files": scan.get("package_files", []),
    }
    fetched_docs = {"saved_files": []}
    project_desc = project_description or "Generated by setup pipeline"

    # 2) Initialize external_context and symlink
    try:
        ecm = ExternalContextManager(wd)
        ecm.initialize_context_structure()
    except Exception as e:
        if dbg:
            dbg.warning(f"pipeline_external_context_init_failed: {e}")

    # 3) Run Agentic Dependency Resolver (TinyAgent) to fetch docs
    try:
        import asyncio
        resolver = AgenticDependencyResolver(
            project_path=str(wd),
            config_manager=config_manager,
            ui_callback=None,
            storage_manager=None,
            textual_ui_callback=textual_ui_callback,
        )
        if dbg:
            dbg.info("pipeline_resolver_start", deps=len(detected["dependencies"]))
        docs_result = asyncio.run(resolver.run())
        if dbg:
            dbg.info("pipeline_resolver_done", success=docs_result.get("success"))

        # Normalize to saved_files for downstream generators
        if docs_result.get("success"):
            file_names = docs_result.get("file_names") or []
            documentation_fetched = docs_result.get("documentation_fetched", {})
            saved_files = []
            if file_names:
                for name in file_names:
                    dep = name.replace(".md", "")
                    saved_files.append({"dependency": dep, "filename": name})
            elif documentation_fetched:
                for fi in documentation_fetched.get("saved_files", []):
                    saved_files.append({
                        "dependency": fi.get("name", "unknown"),
                        "filename": Path(fi.get("path", "")).name or f"{fi.get('name','unknown')}.md",
                    })
            fetched_docs = {"saved_files": saved_files}
    except Exception as e:
        if dbg:
            dbg.warning(f"pipeline_resolver_failed: {e}")

    # 4) Contributor guide
    try:
        from .agents_contributor_agent import generate_contributor_guide
        contributor_text = generate_contributor_guide(wd, logger=dbg)
    except Exception as e:
        contributor_text = ""
        if dbg:
            dbg.warning(f"pipeline_contributor_generation_failed: {e}")

    # 5) Write JUNO.md and IDE docs (prepend contributor)
    generate_and_write_juno_md(
        workdir=wd,
        project_description=project_desc,
        selected_editor=editor_display,
        ai_analysis="",
        detected_deps=detected,
        fetched_docs=fetched_docs,
        logger=dbg,
    )

    if editor_display.lower() in ("claude code", "claude_code"):
        gen = AgentsMdGenerator(logger=dbg)
        content = gen.generate_content(
            AgentsMdInputs(
                workdir=wd,
                ide_name="Claude Code",
                project_description=project_desc,
                ai_analysis="",
                detected_deps=detected,
                fetched_docs=fetched_docs,
            )
        )
        merged_claude = (contributor_text.strip() + "\n\n" + content) if contributor_text else content
        (wd / "CLAUDE.md").write_text(merged_claude, encoding="utf-8")
        # Also write AGENTS.md with the same content for universal discovery
        (wd / "AGENTS.md").write_text(merged_claude, encoding="utf-8")
    else:
        gen_path = generate_and_write_agents_md(
            workdir=wd,
            ide_name=editor_display,
            project_description=project_desc,
            ai_analysis="",
            detected_deps=detected,
            fetched_docs=fetched_docs,
            logger=dbg,
        )
        base = gen_path.read_text(encoding="utf-8") if gen_path and gen_path.exists() else ""
        merged = (contributor_text.strip() + "\n\n" + base) if contributor_text else base
        (wd / "AGENTS.md").write_text(merged, encoding="utf-8")

    # 6) Install MCP for selected editor
    try:
        raw = (editor_display or "").strip().lower()
        if raw in ("claude code", "claude_code"):
            editor_key = "claude_code"
        elif raw in ("cursor",):
            editor_key = "cursor"
        elif raw in ("windsurf",):
            editor_key = "windsurf"
        elif raw in ("vscode", "vs code", "visual studio code"):
            editor_key = "vscode"
        else:
            editor_key = raw.replace(" ", "_") or "cursor"

        installer = MCPInstaller(project_dir=wd)
        ok = installer.install_mcp_servers(editor_key, Path(wd))
        if dbg:
            dbg.info("pipeline_mcp_installed", editor=editor_key, success=ok)
    except MCPInstallationError as e:
        ok = False
        if dbg:
            dbg.warning(f"pipeline_mcp_install_error: {e}")
    except Exception as e:
        ok = False
        if dbg:
            dbg.warning(f"pipeline_mcp_install_unexpected: {e}")

    # 7) Persist editor and MCP flag
    try:
        cfg = config_manager.load_config()
        cfg.editor = editor_display
        cfg.mcp_server_installed = bool(ok)
        # Mark setup as completed for unified pipeline
        try:
            setattr(cfg, 'setup_completed', True)
        except Exception:
            pass
        # Persist project description if provided
        try:
            if project_description:
                setattr(cfg, 'project_description', project_desc)
        except Exception:
            pass
        config_manager.save_config(cfg)
        if dbg:
            dbg.info("pipeline_persisted_config", editor=editor_display, mcp=bool(ok))
        (wd / ".juno_config.json").write_text('{"editor": "' + editor_display + '"}', encoding="utf-8")
    except Exception as e:
        if dbg:
            dbg.warning(f"pipeline_persist_failed: {e}")

    # 8) Verify and (optionally) write report
    import asyncio
    verifier = VerifyAgent(wd, project_name=wd.name, logger=dbg)
    out = asyncio.run(verifier.run(skip_external_calls=False))
    if report_file:
        try:
            Path(report_file).write_text(out.report, encoding="utf-8")
        except Exception as e:
            if dbg:
                dbg.warning(f"pipeline_report_write_failed: {e}")

    # Return summary
    return {
        "success": True,
        "pass": sum(1 for r in out.results if r.status == "PASS"),
        "fail": sum(1 for r in out.results if r.status == "FAIL"),
        "warn": sum(1 for r in out.results if r.status == "WARN"),
        "info": sum(1 for r in out.results if r.status == "INFO"),
    }
