"""Intelligent Agent Registry System.

Unifies access to crackerjack agents, user agents, and system agents with smart
prioritization and capability mapping.
"""

import logging
import typing as t
from dataclasses import dataclass
from enum import Enum
from pathlib import Path

from crackerjack.agents.base import SubAgent, agent_registry


class AgentSource(Enum):
    """Source of an agent."""

    CRACKERJACK = "crackerjack"
    USER = "user"
    SYSTEM = "system"


class AgentCapability(Enum):
    """Agent capability categories."""

    ARCHITECTURE = "architecture"
    REFACTORING = "refactoring"
    TESTING = "testing"
    SECURITY = "security"
    PERFORMANCE = "performance"
    DOCUMENTATION = "documentation"
    FORMATTING = "formatting"
    DEBUGGING = "debugging"
    CODE_ANALYSIS = "code_analysis"
    PROJECT_MANAGEMENT = "project_management"


@dataclass
class AgentMetadata:
    """Metadata about an agent."""

    name: str
    source: AgentSource
    capabilities: set[AgentCapability]
    priority: int  # Higher is better
    confidence_factor: float  # Multiplier for confidence scores
    description: str
    model: str | None = None
    tags: list[str] | None = None


@dataclass
class RegisteredAgent:
    """A registered agent with metadata and access."""

    metadata: AgentMetadata
    agent: SubAgent | None = None  # None for user/system agents
    agent_path: Path | None = None  # Path to user agent file
    subagent_type: str | None = None  # For system agents


class AgentRegistry:
    """Registry of all available agents across all systems."""

    def __init__(self) -> None:
        self.logger = logging.getLogger(__name__)
        self._agents: dict[str, RegisteredAgent] = {}
        self._capability_map: dict[AgentCapability, list[str]] = {}
        self._user_agent_cache: dict[str, dict[str, t.Any]] = {}

    async def initialize(self) -> None:
        """Initialize the registry with all available agents."""
        self.logger.info("Initializing Intelligent Agent Registry")

        # Load agents in priority order
        await self._register_crackerjack_agents()
        await self._register_user_agents()
        await self._register_system_agents()

        # Build capability mapping
        self._build_capability_map()

        self.logger.info(
            f"Registry initialized with {len(self._agents)} agents: "
            f"{len([a for a in self._agents.values() if a.metadata.source == AgentSource.CRACKERJACK])} crackerjack, "
            f"{len([a for a in self._agents.values() if a.metadata.source == AgentSource.USER])} user, "
            f"{len([a for a in self._agents.values() if a.metadata.source == AgentSource.SYSTEM])} system"
        )

    async def _register_crackerjack_agents(self) -> None:
        """Register built-in crackerjack agents."""
        self.logger.debug("Registering crackerjack agents")

        # Get all registered crackerjack agents
        from crackerjack.agents.base import AgentContext

        # Create a dummy context to get agents
        context = AgentContext(
            project_path=Path.cwd(),
        )

        crackerjack_agents = agent_registry.create_all(context)

        for agent in crackerjack_agents:
            capabilities = self._infer_capabilities_from_agent(agent)

            metadata = AgentMetadata(
                name=agent.name,
                source=AgentSource.CRACKERJACK,
                capabilities=capabilities,
                priority=100,  # Highest priority
                confidence_factor=1.0,
                description=f"Built-in crackerjack {agent.__class__.__name__}",
            )

            registered = RegisteredAgent(
                metadata=metadata,
                agent=agent,
            )

            self._agents[agent.name] = registered
            self.logger.debug(f"Registered crackerjack agent: {agent.name}")

    async def _register_user_agents(self) -> None:
        """Register user agents from ~/.claude/agents/."""
        self.logger.debug("Registering user agents")

        user_agents_dir = Path.home() / ".claude" / "agents"
        if not user_agents_dir.exists():
            self.logger.debug("No user agents directory found")
            return

        for agent_file in user_agents_dir.glob("*.md"):
            try:
                agent_data = await self._parse_user_agent_file(agent_file)
                if agent_data:
                    capabilities = self._infer_capabilities_from_user_agent(agent_data)

                    metadata = AgentMetadata(
                        name=agent_data["name"],
                        source=AgentSource.USER,
                        capabilities=capabilities,
                        priority=80,  # Second priority
                        confidence_factor=0.9,
                        description=agent_data.get("description", "User agent"),
                        model=agent_data.get("model"),
                        tags=agent_data.get("tags", []),
                    )

                    registered = RegisteredAgent(
                        metadata=metadata,
                        agent_path=agent_file,
                    )

                    self._agents[agent_data["name"]] = registered
                    self._user_agent_cache[agent_data["name"]] = agent_data
                    self.logger.debug(f"Registered user agent: {agent_data['name']}")

            except Exception as e:
                self.logger.warning(f"Failed to parse user agent {agent_file}: {e}")

    async def _register_system_agents(self) -> None:
        """Register built-in system agents from Task tool."""
        self.logger.debug("Registering system agents")

        # Known system agents from Task tool
        system_agents = [
            (
                "general-purpose",
                "General-purpose agent for researching complex questions",
            ),
            (
                "statusline-setup",
                "Configure the user's Claude Code status line setting",
            ),
            ("output-style-setup", "Create a Claude Code output style"),
        ]

        for agent_name, description in system_agents:
            capabilities = self._infer_capabilities_from_system_agent(
                agent_name, description
            )

            metadata = AgentMetadata(
                name=agent_name,
                source=AgentSource.SYSTEM,
                capabilities=capabilities,
                priority=60,  # Lowest priority
                confidence_factor=0.7,
                description=description,
            )

            registered = RegisteredAgent(
                metadata=metadata,
                subagent_type=agent_name,
            )

            self._agents[agent_name] = registered
            self.logger.debug(f"Registered system agent: {agent_name}")

    async def _parse_user_agent_file(self, agent_file: Path) -> dict[str, t.Any] | None:
        """Parse a user agent markdown file."""
        try:
            content = agent_file.read_text(encoding="utf-8")
            return self._extract_agent_data_from_content(content)
        except Exception as e:
            self.logger.error(f"Error parsing agent file {agent_file}: {e}")
            return None

    def _extract_agent_data_from_content(self, content: str) -> dict[str, t.Any] | None:
        """Extract agent data from file content."""
        if not content.startswith("---\n"):
            return None

        lines = content.split("\n")
        yaml_end = self._find_yaml_end_marker(lines)

        if yaml_end == -1:
            return None

        return self._build_agent_data(lines, yaml_end)

    def _find_yaml_end_marker(self, lines: list[str]) -> int:
        """Find the end marker for YAML frontmatter."""
        for i, line in enumerate(lines[1:], 1):
            if line == "---":
                return i
        return -1

    def _build_agent_data(self, lines: list[str], yaml_end: int) -> dict[str, t.Any]:
        """Build agent data from parsed lines."""
        yaml_lines = lines[1:yaml_end]
        agent_data = {}

        for line in yaml_lines:
            if ":" in line:
                key, value = line.split(":", 1)
                agent_data[key.strip()] = value.strip()

        agent_data["content"] = "\n".join(lines[yaml_end + 1 :])
        return agent_data

    def _infer_capabilities_from_agent(self, agent: SubAgent) -> set[AgentCapability]:
        """Infer capabilities from a crackerjack agent."""
        capabilities = set()

        # Map agent class names to capabilities
        class_name = agent.__class__.__name__.lower()

        if "architect" in class_name:
            capabilities.update(
                {AgentCapability.ARCHITECTURE, AgentCapability.CODE_ANALYSIS}
            )
        if "refactor" in class_name:
            capabilities.add(AgentCapability.REFACTORING)
        if "test" in class_name:
            capabilities.add(AgentCapability.TESTING)
        if "security" in class_name:
            capabilities.add(AgentCapability.SECURITY)
        if "performance" in class_name:
            capabilities.add(AgentCapability.PERFORMANCE)
        if "documentation" in class_name or "doc" in class_name:
            capabilities.add(AgentCapability.DOCUMENTATION)
        if "format" in class_name:
            capabilities.add(AgentCapability.FORMATTING)
        if "import" in class_name:
            capabilities.add(AgentCapability.CODE_ANALYSIS)
        if "dry" in class_name:
            capabilities.add(AgentCapability.REFACTORING)

        # Fallback to general code analysis
        if not capabilities:
            capabilities.add(AgentCapability.CODE_ANALYSIS)

        return capabilities

    def _infer_capabilities_from_user_agent(
        self, agent_data: dict[str, t.Any]
    ) -> set[AgentCapability]:
        """Infer capabilities from user agent metadata."""
        capabilities = set()

        name = agent_data.get("name", "").lower()
        description = agent_data.get("description", "").lower()
        content = agent_data.get("content", "").lower()

        text = f"{name} {description} {content}"

        # Keyword mapping
        keyword_map = {
            AgentCapability.ARCHITECTURE: [
                "architect",
                "design",
                "structure",
                "pattern",
            ],
            AgentCapability.REFACTORING: ["refactor", "clean", "improve", "optimize"],
            AgentCapability.TESTING: ["test", "pytest", "coverage", "mock"],
            AgentCapability.SECURITY: ["security", "secure", "vulnerability", "audit"],
            AgentCapability.PERFORMANCE: [
                "performance",
                "speed",
                "optimize",
                "efficient",
            ],
            AgentCapability.DOCUMENTATION: ["document", "readme", "comment", "explain"],
            AgentCapability.FORMATTING: ["format", "style", "lint", "ruff"],
            AgentCapability.DEBUGGING: ["debug", "fix", "error", "troubleshoot"],
            AgentCapability.CODE_ANALYSIS: ["analyze", "review", "inspect", "examine"],
            AgentCapability.PROJECT_MANAGEMENT: [
                "project",
                "manage",
                "organize",
                "workflow",
            ],
        }

        for capability, keywords in keyword_map.items():
            if any(keyword in text for keyword in keywords):
                capabilities.add(capability)

        # Fallback
        if not capabilities:
            capabilities.add(AgentCapability.CODE_ANALYSIS)

        return capabilities

    def _infer_capabilities_from_system_agent(
        self, name: str, description: str
    ) -> set[AgentCapability]:
        """Infer capabilities from system agent."""
        capabilities = set()

        text = f"{name} {description}".lower()

        if "general" in text or "research" in text:
            capabilities.add(AgentCapability.CODE_ANALYSIS)
        if "statusline" in text or "setup" in text:
            capabilities.add(AgentCapability.PROJECT_MANAGEMENT)
        if "output" in text or "style" in text:
            capabilities.add(AgentCapability.FORMATTING)

        # Fallback
        if not capabilities:
            capabilities.add(AgentCapability.CODE_ANALYSIS)

        return capabilities

    def _build_capability_map(self) -> None:
        """Build mapping from capabilities to agent names."""
        self._capability_map.clear()

        for agent_name, registered_agent in self._agents.items():
            for capability in registered_agent.metadata.capabilities:
                if capability not in self._capability_map:
                    self._capability_map[capability] = []
                self._capability_map[capability].append(agent_name)

        # Sort by priority within each capability
        for agent_names in self._capability_map.values():
            agent_names.sort(
                key=lambda name: self._agents[name].metadata.priority, reverse=True
            )

    def get_agents_by_capability(
        self, capability: AgentCapability
    ) -> list[RegisteredAgent]:
        """Get agents that have a specific capability, sorted by priority."""
        agent_names = self._capability_map.get(capability, [])
        return [self._agents[name] for name in agent_names]

    def get_agent_by_name(self, name: str) -> RegisteredAgent | None:
        """Get a specific agent by name."""
        return self._agents.get(name)

    def list_all_agents(self) -> list[RegisteredAgent]:
        """List all registered agents, sorted by priority."""
        agents = list(self._agents.values())
        agents.sort(key=lambda a: a.metadata.priority, reverse=True)
        return agents

    def get_agent_stats(self) -> dict[str, t.Any]:
        """Get statistics about registered agents."""
        stats: dict[str, t.Any] = {
            "total_agents": len(self._agents),
            "by_source": {},
            "by_capability": {},
        }

        # Count by source
        for source in AgentSource:
            count = len(
                [a for a in self._agents.values() if a.metadata.source == source]
            )
            stats["by_source"][source.value] = count

        # Count by capability
        for capability in AgentCapability:
            count = len(self._capability_map.get(capability, []))
            stats["by_capability"][capability.value] = count

        return stats


# Global registry instance
agent_registry_instance = AgentRegistry()


async def get_agent_registry() -> AgentRegistry:
    """Get the initialized agent registry."""
    if not agent_registry_instance._agents:
        await agent_registry_instance.initialize()
    return agent_registry_instance
