# AgenticlyPay

A Python library for processing agentic payments (ACP, AP2, x402) via Stripe. Built for agentic developers who need automated payment processing, monthly Wise payouts, and tax compliance.

## Features

- **Multi-Protocol Support**: Process payments using ACP (Agentic Commerce Protocol), AP2 (Agent Payments Protocol), or x402 protocols
- **Stripe Connect Integration**: Seamless onboarding and account management for developers
- **Automated Wise Payouts**: Monthly automated payouts via Wise to your email address (international support)
- **Tax Compliance**: Automatic 1099 form generation and filing
- **Transparent Pricing**: 6.5% + $0.30 per transaction
- **Email-Only Configuration**: No API keys needed - just use your email address

## Important: Email Requirements

**Your email address** is used for monthly payouts via Wise. Wise supports international transfers, so you can use any email address regardless of location.

## Installation

```bash
pip install agenticlypay
```

## Quick Start

### Basic Usage

```python
from agenticlypay import AgenticlyPayClient

# Initialize the client (no API keys needed!)
client = AgenticlyPayClient(base_url="https://api.agenticlypay.com")  # Optional, defaults to production

# Create a developer account
account = client.create_account(
    email="developer@example.com",
    country="US"
)

# Process a payment (AUTO protocol)
# IMPORTANT: Include your email for usage tracking
payment = client.process_payment(
    email="developer@example.com",
    protocol="AUTO",
    amount=10000,  # $100.00 in cents
    developer_account_id=account["account"]["account_id"],
    currency="usd",
    description="Payment for service"
    # Include `mandate` for AP2 or `resource_url` for x402
)
```

### Using Convenience Wrappers

```python
from agenticlypay import PaymentProcessor, ConnectManager

# Initialize components (no API keys needed!)
payment_processor = PaymentProcessor()
connect_manager = ConnectManager()

# Create a developer account
account = connect_manager.create_developer_account(
    email="developer@example.com",
    country="US"
)

# Process a payment
result = payment_processor.process_payment(
    email="developer@example.com",  # Required
    protocol="AUTO",
    amount=10000,  # $100.00 in cents
    developer_account_id=account["account_id"],
    currency="usd"
)
```

### ACP Payment Example

```python
from agenticlypay import AgenticlyPayClient

client = AgenticlyPayClient()

result = client.process_payment(
    email="developer@example.com",  # Required
    protocol="ACP",
    amount=10000,  # $100.00 in cents
    currency="usd",
    developer_account_id="acct_xxxxx",
    description="Payment for service"
)
```

**Note**: Monthly payouts are automatically sent to your email address via Wise. Wise supports international transfers, so you can use any email address.

### AP2 Payment Example

```python
result = client.process_payment(
    email="developer@example.com",  # Required
    protocol="AP2",
    amount=10000,
    currency="usd",
    developer_account_id="acct_xxxxx",
    mandate={
        "agent_id": "agent_123",
        "user_id": "user_456",
        "permissions": ["create_payment", "complete_purchase"],
        "expires_at": 1735689600,
        "mandate_id": "mandate_789"
    }
)
```

### x402 Payment Example

```python
result = client.process_payment(
    email="developer@example.com",  # Required
    protocol="x402",
    amount=10000,
    currency="usd",
    developer_account_id="acct_xxxxx",
    resource_url="/api/data/endpoint"
)
```

### Complete Account Setup Example

```python
from agenticlypay import AgenticlyPayClient

client = AgenticlyPayClient()

# 1. Create account
account = client.create_account(
    email="developer@example.com",
    country="US"
)

# 2. Create onboarding link
onboarding = client.create_onboarding_link(
    account_id=account["account"]["account_id"],
    refresh_url="https://yourapp.com/reauth",
    return_url="https://yourapp.com/success",
    email="developer@example.com"  # Required
)

# 3. Redirect user to onboarding.url
print(f"Onboarding URL: {onboarding['onboarding_link']['url']}")

# 4. Process payments
payment = client.process_payment(
    email="developer@example.com",  # Required
    protocol="AUTO",
    amount=10000,
    developer_account_id=account["account"]["account_id"],
    currency="usd"
)
```

## Monthly Wise Payouts

Monthly payouts are automatically sent via Wise to the email address associated with your developer account. Wise supports international transfers, so you can use any email address regardless of location.

## API Reference

### AgenticlyPayClient

The main client class for interacting with the AgenticlyPay API.

#### Methods

- `create_account(email, country="US", metadata=None)` - Create a developer account
- `get_account(account_id, email)` - Get account status
- `create_onboarding_link(account_id, refresh_url, return_url, email)` - Create onboarding link
- `configure_payout_schedule(account_id, interval, email, monthly_anchor=None, weekly_anchor=None)` - Configure payout schedule
- `process_payment(email, protocol, amount, developer_account_id, currency="usd", ...)` - Process a payment
- `confirm_payment(email, protocol, payment_id, payment_method=None)` - Confirm a payment
- `get_payment_status(email, protocol, payment_id)` - Get payment status
- `get_fee(email, amount)` - Calculate fee
- `get_monthly_earnings(email, account_id, year, month)` - Get monthly earnings
- `create_transfer(email, developer_account_id, amount, currency="usd", reference=None)` - Create manual transfer
- `get_annual_earnings(email, account_id, year)` - Get annual earnings for tax reporting

All methods require your email address for usage tracking and payout processing.

## Error Handling

```python
from agenticlypay import AgenticlyPayClient, AgenticlyPayError

client = AgenticlyPayClient()

try:
    account = client.create_account(email="developer@example.com", country="US")
except AgenticlyPayError as e:
    print(f"Error: {e}")
    print(f"Status Code: {e.status_code}")
```

## License

MIT License

## Support

For issues and questions, please visit our GitHub repository or contact support@agenticlypay.com
