"""Unified payment processing interface."""

from typing import Dict, Any, Optional, Literal
from agenticlypay.client import AgenticlyPayClient


class PaymentProcessor:
    """Unified payment processor supporting ACP, AP2, and x402 protocols (wrapper around AgenticlyPayClient)."""

    def __init__(self, base_url: str = "https://api.agenticlypay.com"):
        """Initialize the payment processor."""
        self.client = AgenticlyPayClient(base_url=base_url)

    def process_payment(
        self,
        email: str,
        protocol: Literal["ACP", "AP2", "x402", "AUTO"],
        amount: int,
        developer_account_id: str,
        currency: str = "usd",
        **kwargs,
    ) -> Dict[str, Any]:
        """
        Process a payment using the specified protocol.

        Args:
            email: Developer email address
            protocol: Payment protocol (ACP, AP2, x402, or AUTO)
            amount: Amount in cents
            currency: Currency code (default: usd)
            developer_account_id: Connected account ID for the developer
            **kwargs: Protocol-specific arguments (metadata, description, mandate, resource_url)

        Returns:
            Dictionary with payment information

        Raises:
            ValueError: If protocol is not supported
        """
        result = self.client.process_payment(
            email=email,
            protocol=protocol,
            amount=amount,
            developer_account_id=developer_account_id,
            currency=currency,
            metadata=kwargs.get("metadata"),
            description=kwargs.get("description"),
            mandate=kwargs.get("mandate"),
            resource_url=kwargs.get("resource_url"),
        )
        return result.get("payment", result)

    def confirm_payment(
        self,
        email: str,
        protocol: Literal["AUTO"],
        payment_id: str,
        payment_method: Optional[str] = None,
    ) -> Dict[str, Any]:
        """
        Confirm a payment using the specified protocol.

        Args:
            email: Developer email address
            protocol: Payment protocol
            payment_id: Payment intent ID
            payment_method: Payment method ID (optional)

        Returns:
            Dictionary with payment confirmation result

        Raises:
            ValueError: If protocol is not supported
        """
        result = self.client.confirm_payment(
            email=email,
            protocol=protocol,
            payment_id=payment_id,
            payment_method=payment_method,
        )
        return result.get("payment", result)

    def get_payment_status(
        self,
        email: str,
        protocol: Literal["AUTO"],
        payment_id: str,
    ) -> Dict[str, Any]:
        """
        Get the status of a payment.

        Args:
            email: Developer email address
            protocol: Payment protocol
            payment_id: Payment intent ID

        Returns:
            Dictionary with payment status

        Raises:
            ValueError: If protocol is not supported
        """
        result = self.client.get_payment_status(
            email=email,
            protocol=protocol,
            payment_id=payment_id,
        )
        return result.get("status", result)

    def calculate_fee(self, email: str, amount_cents: int) -> Dict[str, Any]:
        """
        Calculate platform fee for a transaction.

        Args:
            email: Developer email address
            amount_cents: Transaction amount in cents

        Returns:
            Dictionary with fee breakdown
        """
        result = self.client.get_fee(email=email, amount=amount_cents)
        return result.get("fee", result)

    def handle_webhook(
        self, event_data: Dict[str, Any], protocol: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Handle webhook events from Stripe.

        Note: Webhook handling is backend-only and not available via API client.

        Args:
            event_data: Stripe webhook event data
            protocol: Protocol type if known (optional)

        Returns:
            Dictionary with webhook handling result
        """
        raise NotImplementedError("handle_webhook is not available via API client")

