# Zen Token Pricing Strategy & Transparency

## Overview

This document outlines the complete pricing strategy and calculation formulas used by the Zen orchestrator system for token usage and cost transparency. All calculations are compliant with official Claude pricing documentation.

## Core Pricing Engine

The pricing calculations are handled by the `ClaudePricingEngine` located in `/zen/token_transparency/claude_pricing_engine.py`, which serves as the Single Source of Truth (SSOT) for all Claude Code pricing calculations.

## Token Calculation Formulas

### ⚠️ Important: Token Accounting Principles

**CRITICAL**: The `total_tokens` field is comprehensive and includes ALL token types. Cache tokens are NOT added separately to the total - they are already included within it.

- ✅ **Correct**: Display `total_tokens` as the complete count
- ❌ **Incorrect**: Add `total_tokens + cached_tokens` (this double-counts cache tokens)

### Basic Token Types

1. **Input Tokens**: Raw text input sent to Claude models
   - Formula: `usage_data.input_tokens`
   - Counted at full input pricing rate

2. **Output Tokens**: Text generated by Claude models
   - Formula: `usage_data.output_tokens`
   - Counted at output pricing rate (typically 5x input rate)

3. **Cache Read Tokens**: Tokens read from cache
   - Formula: `usage_data.cache_read_tokens`
   - Charged at 10% of input token rate

4. **Cache Creation Tokens**: Tokens used to create cache entries
   - Formula: `usage_data.cache_creation_tokens`
   - Charged at premium rates based on cache duration

5. **Total Tokens**: Sum of all token types (IMPORTANT: This is comprehensive)
   - Formula: `input_tokens + output_tokens + cache_read_tokens + cache_creation_tokens`
   - **Note**: Cache tokens are already included in total_tokens, not added separately

## Cache Pricing Strategy

### Cache Read Cost
```python
cache_read_cost = (cache_read_tokens / 1_000_000) * (model_pricing["input"] * CACHE_READ_MULTIPLIER)
```
Where:
- `CACHE_READ_MULTIPLIER = 0.1` (10% of base input price)

### Cache Creation Cost
```python
cache_creation_cost = (cache_creation_tokens / 1_000_000) * (model_pricing["input"] * cache_multiplier)
```

Where `cache_multiplier` depends on cache duration:
- **5-minute cache**: `CACHE_5MIN_WRITE_MULTIPLIER = 1.25` (25% premium over input rate)
- **1-hour cache**: `CACHE_1HOUR_WRITE_MULTIPLIER = 2.0` (100% premium over input rate)

## Model Pricing Rates (Per Million Tokens)

Current pricing as of 2024-2025:

```python
MODEL_PRICING = {
    "claude-opus-4": {"input": 15.0, "output": 75.0},
    "claude-opus-4.1": {"input": 15.0, "output": 75.0},
    "claude-sonnet-4": {"input": 3.0, "output": 15.0},
    "claude-sonnet-3.7": {"input": 3.0, "output": 15.0},
    "claude-3-5-sonnet": {"input": 3.0, "output": 15.0},
    "claude-haiku-3.5": {"input": 0.8, "output": 4.0},
}
```

## Total Cost Calculation

### Individual Cost Components

```python
# Input cost
input_cost = (input_tokens / 1_000_000) * model_pricing["input"]

# Output cost
output_cost = (output_tokens / 1_000_000) * model_pricing["output"]

# Cache read cost (10% of input rate)
cache_read_cost = (cache_read_tokens / 1_000_000) * (model_pricing["input"] * 0.1)

# Cache creation cost (varies by cache type)
cache_creation_cost = (cache_creation_tokens / 1_000_000) * (model_pricing["input"] * cache_multiplier)

# Tool cost (charged at input rate)
tool_cost = (tool_tokens / 1_000_000) * model_pricing["input"]

# Total cost
total_cost = input_cost + output_cost + cache_read_cost + cache_creation_cost + tool_cost
```

## Tool Pricing Strategy

### Tool Usage Tracking
- Each tool call is tracked individually by tool name
- Token usage per tool is recorded when available
- Tool tokens are charged at the same rate as input tokens

### Tool Cost Calculation
```python
tool_cost = 0.0
if tool_tokens:
    for tool_name, tokens in tool_tokens.items():
        tool_cost += (tokens / 1_000_000) * model_pricing["input"]
```

### Special Tool Pricing
```python
TOOL_PRICING = {
    "web_search": 10.0,  # $10 per 1000 searches (if applicable)
    "web_fetch": 0.0,    # No additional charge
    "default": 0.0       # Most tools have no additional charge
}
```

## Model-Based Cost Tracking

### Why Model Detection Matters for Your Budget

The Model column in Zen's status display isn't just informational - it's critical for accurate cost tracking and budget management. Here's why:

**Cost Variations Between Models** (Per Million Tokens):
- **Claude Opus**: $15 input / $75 output - Premium reasoning
- **Claude Sonnet**: $3 input / $15 output - Balanced performance
- **Claude Haiku**: $0.8 input / $4 output - Fast and efficient

**Real-World Example**:
```
Task: "Analyze this complex codebase"
Your Setting: Sonnet ($3/$15 per million)
Actual Model Used: Opus ($15/$75 per million)
Cost Impact: 5x higher than expected!
```

### How Model Detection Protects Your Budget

1. **Accurate Cost Calculations**: Zen uses the actual model for cost calculations, not your configuration
2. **Budget Compliance**: Budget limits are enforced against real costs, preventing overspend
3. **Transparency**: You see exactly which model processed each request
4. **Optimization Insights**: Identify when you're paying premium rates vs. standard rates

### Model Detection in Action

**Status Display Example**:
```
║  Status   Name             Model      Duration  Cost      Budget
║  ✅        simple-task      haiku35    0m30s     $0.002    5% used
║  🏃        complex-analysis opus4      12m15s    $0.875    78% used
║  ✅        code-review      35sonnet   3m45s     $0.045    25% used
```

**What This Tells You**:
- Simple task was efficiently routed to Haiku (cheapest)
- Complex analysis escalated to Opus (most expensive but most capable)
- Code review used Sonnet (balanced cost/performance)

### Cost Optimization Benefits

**Before Model Detection** (assumptions-based costing):
- Budget: $10 for Sonnet tasks
- Reality: Claude used Opus
- Actual Cost: $50 (5x over budget!)
- Result: Budget violations, surprise costs

**With Model Detection** (reality-based costing):
- Budget: $10 for mixed tasks
- Detection: Haiku ($2) + Sonnet ($5) + Opus ($3)
- Actual Cost: $10 (on budget)
- Result: Accurate tracking, no surprises

## Model Detection Strategy

### Automatic Model Detection
The system automatically detects Claude models from API responses:

1. **Primary locations checked**:
   - `response_data.get('model')`
   - `response_data.get('model_name')`
   - `response_data.get('usage', {}).get('model')`
   - `response_data.get('message', {}).get('model')`
   - `response_data.get('metadata', {}).get('model')`

2. **Model name normalization**:
   - Handles various naming formats (claude-opus-4, opus-4.1, etc.)
   - Defaults to `claude-3-5-sonnet` if detection fails

### Cache Type Detection
The system detects cache duration from response metadata:

1. **Cache type indicators**:
   - `response_data.get('cache_type')`
   - `response_data.get('usage', {}).get('cache_type')`
   - `response_data.get('metadata', {}).get('cache_type')`

2. **Cache type mapping**:
   - `"1hour"` or `"60min"` → 1-hour cache (100% premium)
   - `"5min"` or default → 5-minute cache (25% premium)

## Transparency Features

### Cost Breakdown Report
The system generates detailed transparency reports including:

```python
{
    "model_used": "claude-3-5-sonnet",
    "cache_type": "5min",
    "token_breakdown": {
        "input_tokens": 1500,
        "output_tokens": 800,
        "cache_read_tokens": 200,
        "cache_creation_tokens": 100,
        "total_tokens": 2600,
        "tool_tokens": {"Task": 50}
    },
    "cost_breakdown": {
        "input_cost_usd": 0.0045,
        "output_cost_usd": 0.012,
        "cache_read_cost_usd": 0.00006,
        "cache_creation_cost_usd": 0.000375,
        "tool_cost_usd": 0.00015,
        "total_cost_usd": 0.017285
    },
    "pricing_rates": {
        "model_rates": {"input": 3.0, "output": 15.0},
        "cache_read_multiplier": 0.1,
        "cache_write_multiplier": 1.25
    },
    "compliance_info": {
        "pricing_source": "https://docs.claude.com/en/docs/about-claude/pricing",
        "last_updated": "2024-2025",
        "model_detected": true
    }
}
```

### Real-time Cost Tracking
The orchestrator provides real-time cost tracking in status reports:

1. **Total cost across all instances**
2. **Average cost per instance**
3. **Tool usage details with costs**
4. **Cache efficiency metrics**

### Simplified Metrics Display
For user clarity, the status table shows simplified token metrics:

- **Overall**: Total tokens including all types (input + output + cache_read + cache_creation)
- **Tokens**: Core processing tokens only (input + output)
- **Cache**: Cache tokens only (cache_read + cache_creation)

This provides an intuitive breakdown: `Overall = Tokens + Cache`

#### Status Table Format
```
║  Status   Name                           Model      Duration   Overall  Tokens   Cache    Tools
║  ──────── ────────────────────────────── ────────── ────────── ──────── ──────── ──────── ──────
║  ✅        analyze-repository             35sonnet   5m3s       100.5K   1.1K     99.4K    1
```

#### Tool Usage Details Table
The system also provides a detailed breakdown of tool usage:

```
╔═══ TOOL USAGE DETAILS ═══╗
║ Tool Name            Uses     Tokens     Cost ($)   Used By
║ ──────────────────── ──────── ────────── ────────── ───────────────────────────────────
║ Task                 1        50         0.0002     analyze-repository(1 uses, 50 tok)
║ ──────────────────── ──────── ────────── ────────── ───────────────────────────────────
║ TOTAL                1        50         0.0002
╚═══════════════════════════════════════════════════════════════════════════════════════╝
```

**Tool Cost Calculation**: Tool tokens are charged at the input token rate for the respective model (e.g., $3.00 per million tokens for Claude 3.5 Sonnet).

## Budget Management Integration

### Token Budget Enforcement
```python
# Pre-execution budget check
estimated_tokens = config.max_tokens_per_command or 1000
can_run, reason = budget_manager.check_budget(base_command, estimated_tokens)

# Runtime budget tracking
new_tokens = current_billable_tokens - last_known_total_tokens
budget_manager.record_usage(base_command, new_tokens)
```

### Budget Violation Handling
- **Warn mode**: Log warnings but continue execution
- **Block mode**: Terminate instances that exceed budgets
- **Visual indicators**: Progress bars and status updates

## Data Sources & Compliance

### Authoritative Cost Priority
1. **SDK-provided cost** (highest priority when available)
2. **Pricing engine calculation** (standard method)
3. **Fallback calculation** (legacy support)

### Pricing Source Compliance
- **Primary source**: Official Claude pricing documentation
- **Update frequency**: Maintained in sync with Claude releases
- **Validation**: All calculations cross-referenced with official rates

### Message Deduplication
The system prevents double-counting through message ID tracking:
- Each unique message ID is processed only once
- Prevents duplicate token charges from repeated API responses
- Maintains accuracy across streaming and batch responses

## Usage in Zen Orchestrator

### Status Reports
Real-time status reports include:
- Token usage by type (input, output, cached, tools)
- Cost breakdown per instance
- Budget utilization progress bars
- Tool usage details with token costs

### Final Summary
Orchestration completion includes:
- Total tokens across all instances
- Total cost with Claude pricing compliance
- Cache hit rate and efficiency metrics
- Per-instance detailed breakdown table

## Implementation Notes

### Token Parsing Strategy
1. **JSON parsing** (primary method for stream-json format)
2. **Regex fallback** (backward compatibility)
3. **Progressive accumulation** (handles streaming responses)
4. **Maximum value tracking** (prevents double-counting)

### Cache Backward Compatibility
Maintains legacy `cached_tokens` field:
```python
status.cached_tokens = status.cache_read_tokens + status.cache_creation_tokens
```

### Error Handling
- Graceful fallback when pricing engine unavailable
- Default model assumptions when detection fails
- Comprehensive logging for debugging

## Future Enhancements

### Planned Features
- Historical cost trend analysis
- Budget optimization recommendations
- Advanced caching strategy suggestions
- Multi-currency cost display
- Custom pricing override capabilities

### Integration Roadmap
- Advanced reporting dashboards
- Cost forecasting and predictions
- Team and project cost allocation

---

**Reference**: This pricing strategy is based on the Claude Pricing Engine implementation in `/zen/token_transparency/claude_pricing_engine.py` and is designed to maintain full compliance with official Claude pricing documentation.

**Last Updated**: 2024-2025
**Compliance Source**: https://docs.claude.com/en/docs/about-claude/pricing