"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamEncryption = exports.Stream = void 0;
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const kinesis_generated_1 = require("./kinesis.generated");
const READ_OPERATIONS = [
    'kinesis:DescribeStreamSummary',
    'kinesis:GetRecords',
    'kinesis:GetShardIterator',
    'kinesis:ListShards',
    'kinesis:SubscribeToShard',
];
const WRITE_OPERATIONS = [
    'kinesis:ListShards',
    'kinesis:PutRecord',
    'kinesis:PutRecords',
];
/**
 * Represents a Kinesis Stream.
 */
class StreamBase extends core_1.Resource {
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, ...READ_OPERATIONS);
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...WRITE_OPERATIONS);
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncrypt(grantee);
        return ret;
    }
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...Array.from(new Set([...READ_OPERATIONS, ...WRITE_OPERATIONS])));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(grantee);
        return ret;
    }
    /**
     * Grant the indicated permissions on this stream to the given IAM principal (Role/Group/User).
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.streamArn],
            scope: this,
        });
    }
}
/**
 * A Kinesis stream. Can be encrypted with a KMS key.
 */
class Stream extends StreamBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.streamName,
        });
        const shardCount = props.shardCount || 1;
        const retentionPeriodHours = (_b = (_a = props.retentionPeriod) === null || _a === void 0 ? void 0 : _a.toHours()) !== null && _b !== void 0 ? _b : 24;
        if (!core_1.Token.isUnresolved(retentionPeriodHours)) {
            if (retentionPeriodHours < 24 || retentionPeriodHours > 168) {
                throw new Error(`retentionPeriod must be between 24 and 168 hours. Received ${retentionPeriodHours}`);
            }
        }
        const { streamEncryption, encryptionKey } = this.parseEncryption(props);
        this.stream = new kinesis_generated_1.CfnStream(this, 'Resource', {
            name: this.physicalName,
            retentionPeriodHours,
            shardCount,
            streamEncryption,
        });
        this.streamArn = this.getResourceArnAttribute(this.stream.attrArn, {
            service: 'kinesis',
            resource: 'stream',
            resourceName: this.physicalName,
        });
        this.streamName = this.getResourceNameAttribute(this.stream.ref);
        this.encryptionKey = encryptionKey;
    }
    /**
     * Import an existing Kinesis Stream provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param streamArn Stream ARN (i.e. arn:aws:kinesis:<region>:<account-id>:stream/Foo)
     */
    static fromStreamArn(scope, id, streamArn) {
        return Stream.fromStreamAttributes(scope, id, { streamArn });
    }
    /**
     * Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties
     */
    static fromStreamAttributes(scope, id, attrs) {
        class Import extends StreamBase {
            constructor() {
                super(...arguments);
                this.streamArn = attrs.streamArn;
                this.streamName = core_1.Stack.of(scope).parseArn(attrs.streamArn).resourceName;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _a;
        // if encryption properties are not set, default to KMS in regions where KMS is available
        if (!props.encryption && !props.encryptionKey) {
            const conditionName = 'AwsCdkKinesisEncryptedStreamsUnsupportedRegions';
            const existing = core_1.Stack.of(this).node.tryFindChild(conditionName);
            // create a single condition for the Stack
            if (!existing) {
                new core_1.CfnCondition(core_1.Stack.of(this), conditionName, {
                    expression: core_1.Fn.conditionOr(core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-north-1'), core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-northwest-1')),
                });
            }
            return {
                streamEncryption: core_1.Fn.conditionIf(conditionName, core_1.Aws.NO_VALUE, { EncryptionType: 'KMS', KeyId: 'alias/aws/kinesis' }),
            };
        }
        // default based on whether encryption key is specified
        const encryptionType = (_a = props.encryption) !== null && _a !== void 0 ? _a : (props.encryptionKey ? StreamEncryption.KMS : StreamEncryption.UNENCRYPTED);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== StreamEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === StreamEncryption.UNENCRYPTED) {
            return {};
        }
        if (encryptionType === StreamEncryption.MANAGED) {
            const encryption = { encryptionType: 'KMS', keyId: 'alias/aws/kinesis' };
            return { streamEncryption: encryption };
        }
        if (encryptionType === StreamEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const streamEncryption = {
                encryptionType: 'KMS',
                keyId: encryptionKey.keyArn,
            };
            return { encryptionKey, streamEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Stream = Stream;
/**
 * What kind of server-side encryption to apply to this stream
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Records in the stream are not encrypted.
     */
    StreamEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    StreamEncryption["KMS"] = "KMS";
    /**
     * Server-side encryption with a master key managed by Amazon Kinesis
     */
    StreamEncryption["MANAGED"] = "MANAGED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
//# sourceMappingURL=data:application/json;base64,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