"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
describe('Kinesis data streams', () => {
    test('default stream', () => {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream');
        expect(stack).toMatchTemplate({
            Resources: {
                MyStream5C050E93: {
                    Type: 'AWS::Kinesis::Stream',
                    Properties: {
                        ShardCount: 1,
                        RetentionPeriodHours: 24,
                        StreamEncryption: {
                            'Fn::If': [
                                'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                {
                                    Ref: 'AWS::NoValue',
                                },
                                {
                                    EncryptionType: 'KMS',
                                    KeyId: 'alias/aws/kinesis',
                                },
                            ],
                        },
                    },
                },
            },
            Conditions: {
                AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                    'Fn::Or': [
                        {
                            'Fn::Equals': [
                                {
                                    Ref: 'AWS::Region',
                                },
                                'cn-north-1',
                            ],
                        },
                        {
                            'Fn::Equals': [
                                {
                                    Ref: 'AWS::Region',
                                },
                                'cn-northwest-1',
                            ],
                        },
                    ],
                },
            },
        });
    }),
        test('multiple default streams only have one condition for encryption', () => {
            const stack = new core_1.Stack();
            new lib_1.Stream(stack, 'MyStream');
            new lib_1.Stream(stack, 'MyOtherStream');
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                    MyOtherStream86FCC9CE: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
            });
        }),
        test('stream from attributes', () => {
            const stack = new core_1.Stack();
            const s = lib_1.Stream.fromStreamAttributes(stack, 'MyStream', {
                streamArn: 'arn:aws:kinesis:region:account-id:stream/stream-name',
            });
            expect(s.streamArn).toEqual('arn:aws:kinesis:region:account-id:stream/stream-name');
        }),
        test('uses explicit shard count', () => {
            const stack = new core_1.Stack();
            new lib_1.Stream(stack, 'MyStream', {
                shardCount: 2,
            });
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 2,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
            });
        }),
        test('uses explicit retention period', () => {
            const stack = new core_1.Stack();
            new lib_1.Stream(stack, 'MyStream', {
                retentionPeriod: core_1.Duration.hours(168),
            });
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 168,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
            });
        }),
        test('retention period must be between 24 and 168 hours', () => {
            expect(() => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriod: core_1.Duration.hours(169),
                });
            }).toThrow(/retentionPeriod must be between 24 and 168 hours. Received 169/);
            expect(() => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriod: core_1.Duration.hours(23),
                });
            }).toThrow(/retentionPeriod must be between 24 and 168 hours. Received 23/);
        }),
        test('uses Kinesis master key if MANAGED encryption type is provided', () => {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Stream(stack, 'MyStream', {
                encryption: lib_1.StreamEncryption.MANAGED,
            });
            // THEN
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                EncryptionType: 'KMS',
                                KeyId: 'alias/aws/kinesis',
                            },
                        },
                    },
                },
            });
        }),
        test('encryption key cannot be supplied with UNENCRYPTED as the encryption type', () => {
            const stack = new core_1.Stack();
            const key = new kms.Key(stack, 'myKey');
            expect(() => {
                new lib_1.Stream(stack, 'MyStream', {
                    encryptionKey: key,
                    encryption: lib_1.StreamEncryption.UNENCRYPTED,
                });
            }).toThrow(/encryptionKey is specified, so 'encryption' must be set to KMS/);
        }),
        test('if a KMS key is supplied, infers KMS as the encryption type', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const key = new kms.Key(stack, 'myKey');
            // WHEN
            new lib_1.Stream(stack, 'myStream', {
                encryptionKey: key,
            });
            // THEN
            expect(stack).toHaveResource('AWS::Kinesis::Stream', {
                ShardCount: 1,
                RetentionPeriodHours: 24,
                StreamEncryption: {
                    EncryptionType: 'KMS',
                    KeyId: {
                        'Fn::GetAtt': ['myKey441A1E73', 'Arn'],
                    },
                },
            });
        }),
        test('auto-creates KMS key if encryption type is KMS but no key is provided', () => {
            const stack = new core_1.Stack();
            new lib_1.Stream(stack, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS,
            });
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStreamKey76F3300E: {
                        Type: 'AWS::KMS::Key',
                        Properties: {
                            Description: 'Created by MyStream',
                            KeyPolicy: {
                                Statement: [
                                    {
                                        Action: [
                                            'kms:Create*',
                                            'kms:Describe*',
                                            'kms:Enable*',
                                            'kms:List*',
                                            'kms:Put*',
                                            'kms:Update*',
                                            'kms:Revoke*',
                                            'kms:Disable*',
                                            'kms:Get*',
                                            'kms:Delete*',
                                            'kms:ScheduleKeyDeletion',
                                            'kms:CancelKeyDeletion',
                                            'kms:GenerateDataKey',
                                            'kms:TagResource',
                                            'kms:UntagResource',
                                        ],
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        'arn:',
                                                        {
                                                            Ref: 'AWS::Partition',
                                                        },
                                                        ':iam::',
                                                        {
                                                            Ref: 'AWS::AccountId',
                                                        },
                                                        ':root',
                                                    ],
                                                ],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                        },
                        DeletionPolicy: 'Retain',
                        UpdateReplacePolicy: 'Retain',
                    },
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            RetentionPeriodHours: 24,
                            ShardCount: 1,
                            StreamEncryption: {
                                EncryptionType: 'KMS',
                                KeyId: {
                                    'Fn::GetAtt': ['MyStreamKey76F3300E', 'Arn'],
                                },
                            },
                        },
                    },
                },
            });
        }),
        test('uses explicit KMS key if encryption type is KMS and a key is provided', () => {
            const stack = new core_1.Stack();
            const explicitKey = new kms.Key(stack, 'ExplicitKey', {
                description: 'Explicit Key',
            });
            new lib_1.Stream(stack, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS,
                encryptionKey: explicitKey,
            });
            expect(stack).toMatchTemplate({
                Resources: {
                    ExplicitKey7DF42F37: {
                        Type: 'AWS::KMS::Key',
                        Properties: {
                            Description: 'Explicit Key',
                            KeyPolicy: {
                                Statement: [
                                    {
                                        Action: [
                                            'kms:Create*',
                                            'kms:Describe*',
                                            'kms:Enable*',
                                            'kms:List*',
                                            'kms:Put*',
                                            'kms:Update*',
                                            'kms:Revoke*',
                                            'kms:Disable*',
                                            'kms:Get*',
                                            'kms:Delete*',
                                            'kms:ScheduleKeyDeletion',
                                            'kms:CancelKeyDeletion',
                                            'kms:GenerateDataKey',
                                            'kms:TagResource',
                                            'kms:UntagResource',
                                        ],
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        'arn:',
                                                        {
                                                            Ref: 'AWS::Partition',
                                                        },
                                                        ':iam::',
                                                        {
                                                            Ref: 'AWS::AccountId',
                                                        },
                                                        ':root',
                                                    ],
                                                ],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                        },
                        DeletionPolicy: 'Retain',
                        UpdateReplacePolicy: 'Retain',
                    },
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            RetentionPeriodHours: 24,
                            ShardCount: 1,
                            StreamEncryption: {
                                EncryptionType: 'KMS',
                                KeyId: {
                                    'Fn::GetAtt': ['ExplicitKey7DF42F37', 'Arn'],
                                },
                            },
                        },
                    },
                },
            });
        }),
        test('grantRead creates and attaches a policy with read only access to Stream and EncryptionKey', () => {
            const stack = new core_1.Stack();
            const stream = new lib_1.Stream(stack, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS,
            });
            const user = new iam.User(stack, 'MyUser');
            stream.grantRead(user);
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStreamKey76F3300E: {
                        Type: 'AWS::KMS::Key',
                        Properties: {
                            KeyPolicy: {
                                Statement: [
                                    {
                                        Action: [
                                            'kms:Create*',
                                            'kms:Describe*',
                                            'kms:Enable*',
                                            'kms:List*',
                                            'kms:Put*',
                                            'kms:Update*',
                                            'kms:Revoke*',
                                            'kms:Disable*',
                                            'kms:Get*',
                                            'kms:Delete*',
                                            'kms:ScheduleKeyDeletion',
                                            'kms:CancelKeyDeletion',
                                            'kms:GenerateDataKey',
                                            'kms:TagResource',
                                            'kms:UntagResource',
                                        ],
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        'arn:',
                                                        {
                                                            Ref: 'AWS::Partition',
                                                        },
                                                        ':iam::',
                                                        {
                                                            Ref: 'AWS::AccountId',
                                                        },
                                                        ':root',
                                                    ],
                                                ],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                    {
                                        Action: 'kms:Decrypt',
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::GetAtt': ['MyUserDC45028B', 'Arn'],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            Description: 'Created by MyStream',
                        },
                        UpdateReplacePolicy: 'Retain',
                        DeletionPolicy: 'Retain',
                    },
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                EncryptionType: 'KMS',
                                KeyId: {
                                    'Fn::GetAtt': ['MyStreamKey76F3300E', 'Arn'],
                                },
                            },
                        },
                    },
                    MyUserDC45028B: {
                        Type: 'AWS::IAM::User',
                    },
                    MyUserDefaultPolicy7B897426: {
                        Type: 'AWS::IAM::Policy',
                        Properties: {
                            PolicyDocument: {
                                Statement: [
                                    {
                                        Action: [
                                            'kinesis:DescribeStreamSummary',
                                            'kinesis:GetRecords',
                                            'kinesis:GetShardIterator',
                                            'kinesis:ListShards',
                                            'kinesis:SubscribeToShard',
                                        ],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                                        },
                                    },
                                    {
                                        Action: 'kms:Decrypt',
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStreamKey76F3300E', 'Arn'],
                                        },
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            PolicyName: 'MyUserDefaultPolicy7B897426',
                            Users: [
                                {
                                    Ref: 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
            });
        }),
        test('grantWrite creates and attaches a policy with write only access to Stream and EncryptionKey', () => {
            const stack = new core_1.Stack();
            const stream = new lib_1.Stream(stack, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS,
            });
            const user = new iam.User(stack, 'MyUser');
            stream.grantWrite(user);
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStreamKey76F3300E: {
                        Type: 'AWS::KMS::Key',
                        Properties: {
                            KeyPolicy: {
                                Statement: [
                                    {
                                        Action: [
                                            'kms:Create*',
                                            'kms:Describe*',
                                            'kms:Enable*',
                                            'kms:List*',
                                            'kms:Put*',
                                            'kms:Update*',
                                            'kms:Revoke*',
                                            'kms:Disable*',
                                            'kms:Get*',
                                            'kms:Delete*',
                                            'kms:ScheduleKeyDeletion',
                                            'kms:CancelKeyDeletion',
                                            'kms:GenerateDataKey',
                                            'kms:TagResource',
                                            'kms:UntagResource',
                                        ],
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        'arn:',
                                                        {
                                                            Ref: 'AWS::Partition',
                                                        },
                                                        ':iam::',
                                                        {
                                                            Ref: 'AWS::AccountId',
                                                        },
                                                        ':root',
                                                    ],
                                                ],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                    {
                                        Action: ['kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*'],
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::GetAtt': ['MyUserDC45028B', 'Arn'],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            Description: 'Created by MyStream',
                        },
                        UpdateReplacePolicy: 'Retain',
                        DeletionPolicy: 'Retain',
                    },
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                EncryptionType: 'KMS',
                                KeyId: {
                                    'Fn::GetAtt': ['MyStreamKey76F3300E', 'Arn'],
                                },
                            },
                        },
                    },
                    MyUserDC45028B: {
                        Type: 'AWS::IAM::User',
                    },
                    MyUserDefaultPolicy7B897426: {
                        Type: 'AWS::IAM::Policy',
                        Properties: {
                            PolicyDocument: {
                                Statement: [
                                    {
                                        Action: ['kinesis:ListShards', 'kinesis:PutRecord', 'kinesis:PutRecords'],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                                        },
                                    },
                                    {
                                        Action: ['kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*'],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStreamKey76F3300E', 'Arn'],
                                        },
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            PolicyName: 'MyUserDefaultPolicy7B897426',
                            Users: [
                                {
                                    Ref: 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
            });
        }),
        test('grantReadWrite creates and attaches a policy with access to Stream and EncryptionKey', () => {
            const stack = new core_1.Stack();
            const stream = new lib_1.Stream(stack, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS,
            });
            const user = new iam.User(stack, 'MyUser');
            stream.grantReadWrite(user);
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStreamKey76F3300E: {
                        Type: 'AWS::KMS::Key',
                        Properties: {
                            Description: 'Created by MyStream',
                            KeyPolicy: {
                                Statement: [
                                    {
                                        Action: [
                                            'kms:Create*',
                                            'kms:Describe*',
                                            'kms:Enable*',
                                            'kms:List*',
                                            'kms:Put*',
                                            'kms:Update*',
                                            'kms:Revoke*',
                                            'kms:Disable*',
                                            'kms:Get*',
                                            'kms:Delete*',
                                            'kms:ScheduleKeyDeletion',
                                            'kms:CancelKeyDeletion',
                                            'kms:GenerateDataKey',
                                            'kms:TagResource',
                                            'kms:UntagResource',
                                        ],
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        'arn:',
                                                        {
                                                            Ref: 'AWS::Partition',
                                                        },
                                                        ':iam::',
                                                        {
                                                            Ref: 'AWS::AccountId',
                                                        },
                                                        ':root',
                                                    ],
                                                ],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                    {
                                        Action: ['kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*'],
                                        Effect: 'Allow',
                                        Principal: {
                                            AWS: {
                                                'Fn::GetAtt': ['MyUserDC45028B', 'Arn'],
                                            },
                                        },
                                        Resource: '*',
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                        },
                        DeletionPolicy: 'Retain',
                        UpdateReplacePolicy: 'Retain',
                    },
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            RetentionPeriodHours: 24,
                            ShardCount: 1,
                            StreamEncryption: {
                                EncryptionType: 'KMS',
                                KeyId: {
                                    'Fn::GetAtt': ['MyStreamKey76F3300E', 'Arn'],
                                },
                            },
                        },
                    },
                    MyUserDC45028B: {
                        Type: 'AWS::IAM::User',
                    },
                    MyUserDefaultPolicy7B897426: {
                        Type: 'AWS::IAM::Policy',
                        Properties: {
                            PolicyDocument: {
                                Statement: [
                                    {
                                        Action: [
                                            'kinesis:DescribeStreamSummary',
                                            'kinesis:GetRecords',
                                            'kinesis:GetShardIterator',
                                            'kinesis:ListShards',
                                            'kinesis:SubscribeToShard',
                                            'kinesis:PutRecord',
                                            'kinesis:PutRecords',
                                        ],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                                        },
                                    },
                                    {
                                        Action: ['kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*'],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStreamKey76F3300E', 'Arn'],
                                        },
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            PolicyName: 'MyUserDefaultPolicy7B897426',
                            Users: [
                                {
                                    Ref: 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
            });
        }),
        test('grantRead creates and associates a policy with read only access to Stream', () => {
            const stack = new core_1.Stack();
            const stream = new lib_1.Stream(stack, 'MyStream');
            const user = new iam.User(stack, 'MyUser');
            stream.grantRead(user);
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                    MyUserDC45028B: {
                        Type: 'AWS::IAM::User',
                    },
                    MyUserDefaultPolicy7B897426: {
                        Type: 'AWS::IAM::Policy',
                        Properties: {
                            PolicyDocument: {
                                Statement: [
                                    {
                                        Action: [
                                            'kinesis:DescribeStreamSummary',
                                            'kinesis:GetRecords',
                                            'kinesis:GetShardIterator',
                                            'kinesis:ListShards',
                                            'kinesis:SubscribeToShard',
                                        ],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                                        },
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            PolicyName: 'MyUserDefaultPolicy7B897426',
                            Users: [
                                {
                                    Ref: 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
            });
        }),
        test('grantWrite creates and attaches a policy with write only access to Stream', () => {
            const stack = new core_1.Stack();
            const stream = new lib_1.Stream(stack, 'MyStream');
            const user = new iam.User(stack, 'MyUser');
            stream.grantWrite(user);
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                    MyUserDC45028B: {
                        Type: 'AWS::IAM::User',
                    },
                    MyUserDefaultPolicy7B897426: {
                        Type: 'AWS::IAM::Policy',
                        Properties: {
                            PolicyDocument: {
                                Statement: [
                                    {
                                        Action: ['kinesis:ListShards', 'kinesis:PutRecord', 'kinesis:PutRecords'],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                                        },
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            PolicyName: 'MyUserDefaultPolicy7B897426',
                            Users: [
                                {
                                    Ref: 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
            });
        }),
        test('grantReadWrite creates and attaches a policy with write only access to Stream', () => {
            const stack = new core_1.Stack();
            const stream = new lib_1.Stream(stack, 'MyStream');
            const user = new iam.User(stack, 'MyUser');
            stream.grantReadWrite(user);
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                    MyUserDC45028B: {
                        Type: 'AWS::IAM::User',
                    },
                    MyUserDefaultPolicy7B897426: {
                        Type: 'AWS::IAM::Policy',
                        Properties: {
                            PolicyDocument: {
                                Statement: [
                                    {
                                        Action: [
                                            'kinesis:DescribeStreamSummary',
                                            'kinesis:GetRecords',
                                            'kinesis:GetShardIterator',
                                            'kinesis:ListShards',
                                            'kinesis:SubscribeToShard',
                                            'kinesis:PutRecord',
                                            'kinesis:PutRecords',
                                        ],
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                                        },
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            PolicyName: 'MyUserDefaultPolicy7B897426',
                            Users: [
                                {
                                    Ref: 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
            });
        }),
        test('grant creates and attaches a policy to Stream which includes supplied permissions', () => {
            const stack = new core_1.Stack();
            const stream = new lib_1.Stream(stack, 'MyStream');
            const user = new iam.User(stack, 'MyUser');
            stream.grant(user, 'kinesis:DescribeStream');
            expect(stack).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                    MyUserDC45028B: {
                        Type: 'AWS::IAM::User',
                    },
                    MyUserDefaultPolicy7B897426: {
                        Type: 'AWS::IAM::Policy',
                        Properties: {
                            PolicyDocument: {
                                Statement: [
                                    {
                                        Action: 'kinesis:DescribeStream',
                                        Effect: 'Allow',
                                        Resource: {
                                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                                        },
                                    },
                                ],
                                Version: '2012-10-17',
                            },
                            PolicyName: 'MyUserDefaultPolicy7B897426',
                            Users: [
                                {
                                    Ref: 'MyUserDC45028B',
                                },
                            ],
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
            });
        }),
        test('cross-stack permissions - no encryption', () => {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream');
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            expect(stackA).toMatchTemplate({
                Resources: {
                    MyStream5C050E93: {
                        Type: 'AWS::Kinesis::Stream',
                        Properties: {
                            ShardCount: 1,
                            RetentionPeriodHours: 24,
                            StreamEncryption: {
                                'Fn::If': [
                                    'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                    {
                                        Ref: 'AWS::NoValue',
                                    },
                                    {
                                        EncryptionType: 'KMS',
                                        KeyId: 'alias/aws/kinesis',
                                    },
                                ],
                            },
                        },
                    },
                },
                Conditions: {
                    AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                        'Fn::Or': [
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-north-1',
                                ],
                            },
                            {
                                'Fn::Equals': [
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    'cn-northwest-1',
                                ],
                            },
                        ],
                    },
                },
                Outputs: {
                    ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD: {
                        Value: {
                            'Fn::GetAtt': ['MyStream5C050E93', 'Arn'],
                        },
                        Export: {
                            Name: 'stackA:ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD',
                        },
                    },
                },
            });
        }),
        test('fails with encryption due to cyclic dependency', () => {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS,
            });
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            expect(() => {
                app.synth();
            }).toThrow(/'stack.' depends on 'stack.'/);
        });
    test('accepts if retentionPeriodHours is a Token', () => {
        const stack = new core_1.Stack();
        const parameter = new core_1.CfnParameter(stack, 'my-retention-period', {
            type: 'Number',
            default: 48,
            minValue: 24,
            maxValue: 168,
        });
        new lib_1.Stream(stack, 'MyStream', {
            retentionPeriod: core_1.Duration.hours(parameter.valueAsNumber),
        });
        expect(stack).toMatchTemplate({
            Parameters: {
                myretentionperiod: {
                    Type: 'Number',
                    Default: 48,
                    MaxValue: 168,
                    MinValue: 24,
                },
            },
            Resources: {
                MyStream5C050E93: {
                    Type: 'AWS::Kinesis::Stream',
                    Properties: {
                        ShardCount: 1,
                        RetentionPeriodHours: {
                            Ref: 'myretentionperiod',
                        },
                        StreamEncryption: {
                            'Fn::If': [
                                'AwsCdkKinesisEncryptedStreamsUnsupportedRegions',
                                {
                                    Ref: 'AWS::NoValue',
                                },
                                {
                                    EncryptionType: 'KMS',
                                    KeyId: 'alias/aws/kinesis',
                                },
                            ],
                        },
                    },
                },
            },
            Conditions: {
                AwsCdkKinesisEncryptedStreamsUnsupportedRegions: {
                    'Fn::Or': [
                        {
                            'Fn::Equals': [
                                {
                                    Ref: 'AWS::Region',
                                },
                                'cn-north-1',
                            ],
                        },
                        {
                            'Fn::Equals': [
                                {
                                    Ref: 'AWS::Region',
                                },
                                'cn-northwest-1',
                            ],
                        },
                    ],
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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