"""
This allows to use global variables inside the FastAPI application using async mode.
# Usage
Just import `g` and then access (set/get) attributes of it:
```python
from your_project.globals import g
g.foo = "foo"
# In some other code
assert g.foo == "foo"
```
Best way to utilize the global `g` in your code is to set the desired
value in a FastAPI dependency, like so:
```python
async def set_global_foo() -> None:
    g.foo = "foo"
@app.get("/test/", dependencies=[Depends(set_global_foo)])
async def test():
    assert g.foo == "foo"
```
# Setup
Add the `GlobalsMiddleware` to your app:
```python
app = fastapi.FastAPI(
    title="Your app API",
)
app.add_middleware(GlobalsMiddleware)  # <-- This line is necessary
```
Then just use it. ;-)
# Default values
You may use `g.set_default("name", some_value)` to set a default value
for a global variable. This default value will then be used instead of `None`
when the variable is accessed before it was set.
Note that default values should only be set at startup time, never
inside dependencies or similar. Otherwise you may run into the issue that
the value was already used any thus have a value of `None` set already, which
would result in the default value not being used.
"""

from collections.abc import Awaitable, Callable
from contextvars import ContextVar, copy_context
from typing import TYPE_CHECKING, Any

from fastapi import Request, Response
from starlette.middleware.base import BaseHTTPMiddleware
from starlette.types import ASGIApp

from .auth import AuthConfigurator, CookieConfig, JWTStrategyConfig
from .config import Config
from .const import (
    COOKIE_CONFIG_KEYS,
    COOKIE_STRATEGY_KEYS,
    DEFAULT_ADMIN_ROLE,
    DEFAULT_PUBLIC_ROLE,
    JWT_STRATEGY_KEYS,
    ROLE_KEYS,
)

if TYPE_CHECKING:
    from .fastapi_react_toolkit import FastAPIReactToolkit
    from .models import User

__all__ = ["g"]


class Globals:
    __slots__ = ("_vars", "_defaults")

    _vars: dict[str, ContextVar]
    _defaults: dict[str, Any]

    # Type annotations for the attributes
    user: "User"
    auth: AuthConfigurator
    config: Config
    admin_role: str
    public_role: str
    is_migrate: bool
    sensitive_data: dict[str, list[str]]
    """
    A dictionary used to store list of sensitive columns for each model that should not be returned in the list and get endpoints. Default is `{"User": ["password", "hashed_password"]}`.
    """
    current_app: "FastAPIReactToolkit"

    def __init__(self) -> None:
        object.__setattr__(self, "_vars", {})
        object.__setattr__(self, "_defaults", {})

    def set_default(self, name: str, default: Any) -> None:
        """Set a default value for a variable."""

        # Ignore if default is already set and is the same value
        if name in self._defaults and default is self._defaults[name]:
            return

        # Ensure we don't have a value set already - the default will have
        # no effect then
        if name in self._vars:
            raise RuntimeError(
                f"Cannot set default as variable {name} was already set",
            )

        # Set the default already!
        self._defaults[name] = default

    def _get_default_value(self, name: str) -> Any:
        """Get the default value for a variable."""

        default = self._defaults.get(name, None)

        return default() if callable(default) else default

    def _ensure_var(self, name: str) -> None:
        """Ensure a ContextVar exists for a variable."""

        if name not in self._vars:
            default = self._get_default_value(name)
            self._vars[name] = ContextVar(f"globals:{name}", default=default)

    def __getattr__(self, name: str) -> Any:
        """Get the value of a variable."""

        self._ensure_var(name)
        return self._vars[name].get()

    def __setattr__(self, name: str, value: Any) -> None:
        """Set the value of a variable."""

        self._ensure_var(name)
        self._vars[name].set(value)


async def globals_middleware_dispatch(
    request: Request,
    call_next: Callable,
) -> Response:
    """Dispatch the request in a new context to allow globals to be used."""

    ctx = copy_context()

    def _call_next() -> Awaitable[Response]:
        return call_next(request)

    return await ctx.run(_call_next)


class GlobalsMiddleware(BaseHTTPMiddleware):  # noqa
    """Middleware to setup the globals context using globals_middleware_dispatch()."""

    def __init__(self, app: ASGIApp) -> None:
        super().__init__(app, globals_middleware_dispatch)


def basic_callback():
    """
    Initializes the configuration settings for the FastAPI React Toolkit.

    This method reads the configuration values from the `g.config` dictionary and sets the corresponding attributes
    in the `g.auth`. It also sets the default values for any missing configuration settings.

    The configuration settings that are read and set include:
    - Authentication keys from `g.auth`
    - Cookie configuration settings
    - Cookie strategy configuration settings
    - JWT strategy configuration settings
    - Role keys

    Note: The `g.config` dictionary should contain the necessary configuration values for the FastAPI React Toolkit.
    """
    auth_keys = g.auth.keys()
    cookie_config: CookieConfig = {}
    cookie_strategy_config: JWTStrategyConfig = {}
    jwt_strategy_config: JWTStrategyConfig = {}
    for key, value in g.config.items():
        lower_key = key.lower()
        if lower_key in auth_keys:
            setattr(g.auth, lower_key, value)
        if lower_key in COOKIE_CONFIG_KEYS:
            cookie_config[lower_key] = value
        if lower_key in COOKIE_STRATEGY_KEYS:
            lower_key = lower_key.replace("cookie_", "")
            cookie_strategy_config[lower_key] = value
        if lower_key in JWT_STRATEGY_KEYS:
            lower_key = lower_key.replace("jwt_", "")
            jwt_strategy_config[lower_key] = value
        if lower_key in ROLE_KEYS:
            lower_key = lower_key.replace("auth_", "")
            setattr(g, lower_key, value)

    if cookie_config:
        g.auth.cookie_config = cookie_config
    if cookie_strategy_config:
        g.auth.cookie_strategy_config = cookie_strategy_config
    if jwt_strategy_config:
        g.auth.jwt_strategy_config = jwt_strategy_config


g = Globals()
g.set_default("auth", AuthConfigurator())
g.set_default("config", Config())
g.set_default("admin_role", DEFAULT_ADMIN_ROLE)
g.set_default("public_role", DEFAULT_PUBLIC_ROLE)
g.set_default("is_migrate", False)
g.set_default("sensitive_data", {"User": ["password", "hashed_password"]})
g.config.add_callback(basic_callback)
