/*
 *  Copyright (C) GridGain Systems. All Rights Reserved.
 *  _________        _____ __________________        _____
 *  __  ____/___________(_)______  /__  ____/______ ____(_)_______
 *  _  / __  __  ___/__  / _  __  / _  / __  _  __ `/__  / __  __ \
 *  / /_/ /  _  /    _  /  / /_/ /  / /_/ /  / /_/ / _  /  _  / / /
 *  \____/   /_/     /_/   \_,__/   \____/   \__,_/  /_/   /_/ /_/
 */

#pragma once

#include "ignite/protocol/sql/column_meta.h"
#include "ignite/odbc/query/query.h"

namespace ignite {

/** Connection forward-declaration. */
class sql_connection;

/**
 * Column metadata.
 */
struct odbc_column_meta {
    /**
     * Default constructor.
     */
    odbc_column_meta() = default;

    /** Label. */
    std::string label;

    /** Schema name. */
    std::optional<std::string> schema;

    /** Table name. */
    std::optional<std::string> table;

    /** Column name. */
    std::optional<std::string> column;

    /** Data type. */
    ignite_type data_type{ignite_type::UNDEFINED};

    /** Nullability. */
    bool nullable{false};

    /** Precision. */
    std::int32_t precision{0};

    /** Scale. */
    std::int32_t scale{0};
};

/**
 * Query.
 */
class column_metadata_query : public query {
public:
    /**
     * Constructor.
     *
     * @param diag Diagnostics collector.
     * @param connection Associated connection.
     * @param schema Schema search pattern.
     * @param table Table search pattern.
     * @param column Column search pattern.
     */
    column_metadata_query(diagnosable_adapter &diag, sql_connection &connection, std::string schema, std::string table,
        std::string column);

    /**
     * Destructor.
     */
    ~column_metadata_query() override = default;

    /**
     * Execute query.
     *
     * @return True on success.
     */
    sql_result execute() override;

    /**
     * Get column metadata.
     *
     * @return Column metadata.
     */
    const protocol::column_meta_vector *get_meta() override { return &m_columns_meta; }

    /**
     * Fetch next result row to application buffers.
     *
     * @param column_bindings Column bindings.
     * @return Operation result.
     */
    sql_result fetch_next_row(column_binding_map &column_bindings) override;

    /**
     * Get data of the specified column in the result set.
     *
     * @param column_idx Column index.
     * @param buffer Buffer to put column data to.
     * @return Operation result.
     */
    sql_result get_column(std::uint16_t column_idx, application_data_buffer &buffer) override;

    /**
     * Close query.
     *
     * @return True on success.
     */
    sql_result close() override;

    /**
     * Check if data is available.
     *
     * @return True if data is available.
     */
    bool is_data_available() const override { return m_cursor != m_meta.end(); }

    /**
     * Get the number of rows affected by the statement.
     *
     * @return Number of rows affected by the statement.
     */
    [[nodiscard]] std::int64_t affected_rows() const override { return 0; }

    /**
     * Move to the next result set.
     *
     * @return Operation result.
     */
    sql_result next_result_set() override { return sql_result::AI_NO_DATA; }

private:
    /**
     * Make get columns metadata requests and use response to set internal state.
     *
     * @return Operation result.
     */
    sql_result make_request_get_columns_meta();

    /** Connection associated with the statement. */
    sql_connection &m_connection;

    /** Schema search pattern. */
    std::string m_schema;

    /** Table search pattern. */
    std::string m_table;

    /** Column search pattern. */
    std::string m_column;

    /** Query executed. */
    bool m_executed{false};

    /** Fetched flag. */
    bool m_fetched{false};

    /** Result set flag. */
    bool m_has_result_set{false};

    /** Fetched metadata. */
    std::vector<odbc_column_meta> m_meta;

    /** Metadata cursor. */
    std::vector<odbc_column_meta>::iterator m_cursor;

    /** Columns metadata. */
    protocol::column_meta_vector m_columns_meta;
};

} // namespace ignite
