# Copyright 2024 The Langfun Authors
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Audio types."""

import functools
from langfun.core.modalities import mime


class Audio(mime.Mime):
  """Represents audio for communicating with language models.

  `lf.Audio` can be initialized from a URI (HTTP/HTTPS URL or local path)
  using `lf.Audio.from_uri()` or from raw bytes using `lf.Audio.from_bytes()`.

  **Example:**

  ```python
  import langfun as lf

  # Load audio from path
  audio = lf.Audio.from_path('/path/to/audio.mp3')

  # Use audio in a prompt
  prompt = lf.Template(
      'What is being said in this audio? {{audio}}', audio=audio
  )
  response = lf.query(prompt, lm=lf.llms.Gemini25Flash())
  print(response)
  ```
  """

  MIME_PREFIX = 'audio'

  @functools.cached_property
  def audio_format(self) -> str:
    return self.mime_type.removeprefix(self.MIME_PREFIX + '/')

  def _mime_control_for(self, uri: str) -> str:
    return f'<audio controls> <source src="{uri}"> </audio>'
