"""
EIA Band Plot Package
=====================

This package provides a convenient function for generating
`5‑year band` plots similar to those used by the U.S. Energy
Information Administration (EIA).  These plots visualize the
historical range, average, and current/prior year values for a
time series, making it easy to see how recent data compare with
the last five years of history.

The primary entry point is :func:`~analysis3054.five_year_plot`.

Example
-------
.. code-block:: python

    import pandas as pd
    from analysis3054 import five_year_plot

    # Suppose `df` has a ``date`` column and one or more value columns
    fig = five_year_plot(date=df['date'], df=df)
    fig.show()
"""

from .plot import five_year_plot
from .ml import ml_forecast, ForecastResult
from .predict import monthly_predictor, MonthlyPredictionResult
from .utils import (
    conditional_column_merge,
    conditional_row_merge,
    nearest_key_merge,
    coalesce_merge,
    rolling_fill,
    data_quality_report,
)
from .estimators import (
    bayesian_linear_estimator,
    BayesianLinearResult,
    gaussian_process_estimator,
    GaussianProcessResult,
    load_based_forecast,
    LoadForecastResult,
)
from .finance import (
    liquidity_adjusted_volatility,
    LiquidityAdjustedVolatilityResult,
    rolling_beta,
    RollingBetaResult,
)
from .visualization import (
    cumulative_return_plot,
    DrawdownResult,
    max_drawdown,
    forecast_plot,
    acf_pacf_plot,
)
from .stats import (
    cross_correlation_plot,
    partial_autocorrelation_plot,
    PCAResult,
    pca_decomposition,
    granger_causality_matrix,
)
from .forecasting import (
    arima_forecast,
    ArimaForecastResult,
    ets_forecast,
    EtsForecastResult,
    var_forecast,
    VarForecastResult,
    auto_arima_forecast,
    AutoArimaForecastResult,
    prophet_forecast,
    ProphetForecastResult,
    markov_switching_forecast,
    MarkovSwitchingForecastResult,
    unobserved_components_forecast,
    UnobservedComponentsForecastResult,
    dynamic_factor_forecast,
    DynamicFactorForecastResult,
    # Newly added advanced forecasting functions
    sarimax_forecast,
    SarimaxForecastResult,
    lstm_forecast,
    LstmForecastResult,
    garch_forecast,
    GarchForecastResult,
    vecm_forecast,
    VecmForecastResult,
    xgboost_forecast,
    XGBoostForecastResult,
    lightgbm_forecast,
    LightGBMForecastResult,
    theta_forecast,
    ThetaForecastResult,
    # Additional advanced forecasting methods
    elastic_net_forecast,
    ElasticNetForecastResult,
    svr_forecast,
    SvrForecastResult,
    tcn_forecast,
    TcnForecastResult,
    bats_forecast,
    BatsForecastResult,
    neuralprophet_forecast,
    NeuralProphetForecastResult,
    catboost_forecast,
    CatBoostForecastResult,
    knn_forecast,
    KnnForecastResult,
    transformer_forecast,
    TransformerForecastResult,
)
from .advanced import (
    harmonic_forecast,
    HarmonicForecastResult,
    ewma_volatility,
    EwmaVolatilityResult,
    monte_carlo_simulation,
    MonteCarloSimulationResult,
    value_at_risk,
    VaRResult,
    cointegration_test,
    CointegrationTestResult,
    spectral_density_plot,
    wavelet_spectrogram,
    kalman_smoother,
    resample_time_series,
    stl_decompose_plot,
)

# Statistical functions
from .statistics import (
    hurst_exponent,
    dfa_exponent,
    rolling_sharpe_ratio,
    sample_entropy,
    higuchi_fractal_dimension,
    RollingSharpeResult,
    rolling_zscore,
    RollingZScoreResult,
    mann_kendall_test,
    MannKendallResult,
    bollinger_bands,
    BollingerBandsResult,
    stationarity_tests,
    StationarityTestResult,
    trend_seasonality_strength,
    TrendSeasonalityStrengthResult,
    box_cox_transform,
    BoxCoxTransformResult,
    seasonal_adjust,
    SeasonalAdjustmentResult,
)

# Regression and correlation utilities
from .regression import (
    ols_regression,
    RegressionResult,
    rolling_correlation,
    RollingCorrelationResult,
    cusum_olsresid_test,
    CusumTestResult,
)

__all__ = [
    "five_year_plot",
    "ml_forecast",
    "ForecastResult",
    "monthly_predictor",
    "MonthlyPredictionResult",
    "harmonic_forecast",
    "HarmonicForecastResult",
    "ewma_volatility",
    "EwmaVolatilityResult",
    "monte_carlo_simulation",
    "MonteCarloSimulationResult",
    "value_at_risk",
    "VaRResult",
    "cointegration_test",
    "CointegrationTestResult",
    "spectral_density_plot",
    "wavelet_spectrogram",
    "kalman_smoother",
    "resample_time_series",
    "stl_decompose_plot",
    # data utilities
    "conditional_column_merge",
    "conditional_row_merge",
    "nearest_key_merge",
    "coalesce_merge",
    "rolling_fill",
    # advanced estimators
    "bayesian_linear_estimator",
    "BayesianLinearResult",
    "gaussian_process_estimator",
    "GaussianProcessResult",
    "load_based_forecast",
    "LoadForecastResult",
    # financial analytics
    "liquidity_adjusted_volatility",
    "LiquidityAdjustedVolatilityResult",
    "rolling_beta",
    "RollingBetaResult",
    # additional utilities and plotting
    "data_quality_report",
    "cumulative_return_plot",
    "DrawdownResult",
    "max_drawdown",
    "forecast_plot",
    "acf_pacf_plot",
    # statistical analysis
    "cross_correlation_plot",
    "partial_autocorrelation_plot",
    "PCAResult",
    "pca_decomposition",
    "granger_causality_matrix",
    # forecasting
    "arima_forecast",
    "ArimaForecastResult",
    "ets_forecast",
    "EtsForecastResult",
    "var_forecast",
    "VarForecastResult",
    "auto_arima_forecast",
    "AutoArimaForecastResult",
    "prophet_forecast",
    "ProphetForecastResult",
    "markov_switching_forecast",
    "MarkovSwitchingForecastResult",
    "unobserved_components_forecast",
    "UnobservedComponentsForecastResult",
    "dynamic_factor_forecast",
    "DynamicFactorForecastResult",
    # Newly added advanced forecasting functions
    "sarimax_forecast",
    "SarimaxForecastResult",
    "lstm_forecast",
    "LstmForecastResult",
    "garch_forecast",
    "GarchForecastResult",
    "vecm_forecast",
    "VecmForecastResult",
    "xgboost_forecast",
    "XGBoostForecastResult",
    "lightgbm_forecast",
    "LightGBMForecastResult",
    "theta_forecast",
    "ThetaForecastResult",
    # newly added advanced forecasting methods
    "elastic_net_forecast",
    "ElasticNetForecastResult",
    "svr_forecast",
    "SvrForecastResult",
    "tcn_forecast",
    "TcnForecastResult",
    "bats_forecast",
    "BatsForecastResult",
    "neuralprophet_forecast",
    "NeuralProphetForecastResult",
    "catboost_forecast",
    "CatBoostForecastResult",
    "knn_forecast",
    "KnnForecastResult",
    "transformer_forecast",
    "TransformerForecastResult",
    "hurst_exponent",
    "dfa_exponent",
    "rolling_sharpe_ratio",
    "RollingSharpeResult",
    "sample_entropy",
    "higuchi_fractal_dimension",
    "rolling_zscore",
    "RollingZScoreResult",
    "mann_kendall_test",
    "MannKendallResult",
    "bollinger_bands",
    "BollingerBandsResult",
    "stationarity_tests",
    "StationarityTestResult",
    "trend_seasonality_strength",
    "TrendSeasonalityStrengthResult",
    "box_cox_transform",
    "BoxCoxTransformResult",
    "seasonal_adjust",
    "SeasonalAdjustmentResult",
    # regression and correlation
    "ols_regression",
    "RegressionResult",
    "rolling_correlation",
    "RollingCorrelationResult",
    "cusum_olsresid_test",
    "CusumTestResult",
]