"""Generator for per-expression compiled evaluators.

This module exposes `ensure_compiled_evaluator(key, expr_ast)` which will
generate a small Cython `.pyx` wrapper for the expression AST, compile it
in-place into `draken/compiled_evaluators/` and import the resulting module.

For safety and simplicity the generator only handles DNF-like expressions
consisting of ORs of AND-clauses where each atom is a column vs literal or
column vs column comparison using equals/greater/less/etc.
"""

import importlib
from pathlib import Path
from typing import Tuple

from opteryx import __version__
from opteryx.draken.evaluators.expression import BinaryExpression
from opteryx.draken.evaluators.expression import ColumnExpression
from opteryx.draken.evaluators.expression import LiteralExpression
from opteryx.third_party.cyan4973.xxhash import hash_bytes

PACKAGE_DIR = Path(__file__).parent / "compiled"
GENERATED_DIR = PACKAGE_DIR
VERSION = hex(hash_bytes(__version__.encode("utf-8")))[-8:]


def has_compiled_evaluator(key: str) -> bool:
    """Check whether a compiled evaluator module exists for key."""
    mod_name = f"opteryx.draken.evaluators.compiled.generated_{VERSION}_{key}"
    try:
        importlib.import_module(mod_name)
        return True
    except ImportError:
        return False


def ensure_compiled_evaluator(key: str, expr_ast) -> Tuple[str, object]:
    """Ensure a compiled evaluator exists for `key` and `expr_ast`.

    Returns (module_name, module_obj).
    """
    mod_name = f"opteryx.draken.evaluators.compiled.generated_{VERSION}_{key}"
    pyx_name = GENERATED_DIR / f"generated_{VERSION}_{key}.pyx"
    so_name = GENERATED_DIR / f"generated_{VERSION}_{key}.cpython-*.so"

    # Check if .so is newer than .pyx
    if pyx_name.exists() and so_name.exists():
        so_files = list(GENERATED_DIR.glob(f"generated_{VERSION}_{key}.cpython-*.so"))
        if so_files and so_files[0].stat().st_mtime >= pyx_name.stat().st_mtime:
            try:
                mod = importlib.import_module(mod_name)
                return mod_name, mod
            except ImportError:
                pass  # Fall through to recompilation

    # Emit a simple .pyx file for the expression.
    pyx_name = GENERATED_DIR / f"generated_{VERSION}_{key}.pyx"
    tmpl = _emit_pyx_for_expr(expr_ast)
    pyx_name.write_text(tmpl)

    # Compile just this specific .pyx file using Cython's API
    try:
        from Cython.Build import cythonize
        from setuptools import Extension

        # Create extension for just this file
        ext = Extension(
            name=mod_name,
            sources=[str(pyx_name)],
            include_dirs=[],
            language="c++",
            extra_compile_args=[
                "-O3",
                "-ffast-math",
                "-march=native",  # Optimize for current CPU
                "-DNDEBUG",  # Disable asserts
            ],
        )

        # Compile in-place
        cythonize(
            [ext],
            force=True,  # Always recompile
            quiet=True,  # Suppress output
            nthreads=2,
        )

        # Build the extension
        from distutils.dist import Distribution

        dist = Distribution({"ext_modules": cythonize([ext])})
        # Don't parse command line - we're being called programmatically
        build_ext_cmd = dist.get_command_obj("build_ext")
        build_ext_cmd.inplace = 1
        build_ext_cmd.finalize_options()
        build_ext_cmd.run()

    except Exception as e:
        raise RuntimeError(f"Failed to compile generated evaluator: {e}") from e

    # Import generated module
    mod = importlib.import_module(mod_name)
    return mod_name, mod


_PYX_TEMPLATE = f"""# cython: language_level=3
# cython: boundscheck=False
# cython: wraparound=False  
# cython: nonecheck=False
# cython: cdivision=True
# cython: initializedcheck=False
# cython: infer_types=False
# cython: embedsignature=False
# cython: c_string_type=bytes
# cython: c_string_encoding=ascii

# ###################################################
# This file is auto-generated. Do not edit directly.
# 
# Generated by opteryx.draken.evaluators.generator
# Opteryx version: {__version__}
# ###################################################

from opteryx.draken.morsels.morsel import Morsel
from opteryx.draken.vectors import BoolMask
from opteryx.draken.compiled import and_mask, or_mask, xor_mask

# Generated code below this point
"""


def _emit_pyx_for_expr(expr_ast) -> str:
    # Very small codegen for DNF: ORs of ANDs where each atom is
    # (col OP literal) or (col OP col) and OP is equals/not_equals/gt/lt.
    # We'll generate a function that:
    #  - obtains columns via morsel.column(b'name')
    #  - calls vector comparison methods which return BoolVector
    #  - extracts ptr.data and length and uses compiled maskops to combine
    # This is intentionally conservative and assumes BoolVector.ptr.data is
    # the bit-packed buffer address.

    # Simple recursive flattening for OR of AND
    def flatten_or(e):
        if isinstance(e, BinaryExpression) and e.operation == "or":
            return flatten_or(e.left) + flatten_or(e.right)
        return [e]

    def flatten_and(e):
        if isinstance(e, BinaryExpression) and e.operation == "and":
            return flatten_and(e.left) + flatten_and(e.right)
        return [e]

    clauses = []
    # top-level could be OR of ANDs or a single clause
    if isinstance(expr_ast, BinaryExpression) and expr_ast.operation == "or":
        parts = flatten_or(expr_ast)
        for p in parts:
            if isinstance(p, BinaryExpression) and p.operation == "and":
                clauses.append(flatten_and(p))
            else:
                clauses.append([p])
    else:
        if isinstance(expr_ast, BinaryExpression) and expr_ast.operation == "and":
            clauses.append(flatten_and(expr_ast))
        else:
            clauses.append([expr_ast])

    # Build imports and header
    header = [_PYX_TEMPLATE]
    body_lines = ["def evaluate(morsel: Morsel):"]
    body_lines.append("    # Acquire comparison masks for each atom")
    body_lines.append("    cdef list masks = []")

    for cidx, clause in enumerate(clauses):
        for aidx, atom in enumerate(clause):
            if not isinstance(atom, BinaryExpression):
                raise ValueError("Unsupported atom in DNF codegen")
            op = atom.operation
            left = atom.left
            right = atom.right
            if isinstance(left, ColumnExpression) and isinstance(right, LiteralExpression):
                col = left.column_name
                lit = right.value
                lit_py = repr(lit)
                body_lines.append("")
                body_lines.append(f"    # atom {cidx}.{aidx}: {col} {op} {lit_py}")
                body_lines.append(f"    vec_{cidx}_{aidx} = morsel.column(b'{col}')")
                # map operation to method name
                method = {
                    "equals": "equals",
                    "not_equals": "not_equals",
                    "greater_than": "greater_than",
                    "greater_than_or_equals": "greater_than_or_equals",
                    "less_than": "less_than",
                    "less_than_or_equals": "less_than_or_equals",
                }.get(op)
                if method is None:
                    raise ValueError("Unsupported comparison op in codegen")
                body_lines.append(f"    mask_{cidx}_{aidx} = vec_{cidx}_{aidx}.{method}({lit_py})")
                body_lines.append(f"    masks.append(mask_{cidx}_{aidx})")
            elif isinstance(left, ColumnExpression) and isinstance(right, ColumnExpression):
                lcol = left.column_name
                rcol = right.column_name
                body_lines.append("")
                body_lines.append(f"    # atom {cidx}.{aidx}: {lcol} {op} {rcol}")
                body_lines.append(f"    vec_{cidx}_{aidx}_l = morsel.column(b'{lcol}')")
                body_lines.append(f"    vec_{cidx}_{aidx}_r = morsel.column(b'{rcol}')")
                method = {
                    "equals": "equals_vector",
                    "not_equals": "not_equals_vector",
                    "greater_than": "greater_than_vector",
                    "greater_than_or_equals": "greater_than_or_equals_vector",
                    "less_than": "less_than_vector",
                    "less_than_or_equals": "less_than_or_equals_vector",
                }.get(op)
                if method is None:
                    raise ValueError("Unsupported comparison op in codegen")
                body_lines.append(
                    f"    mask_{cidx}_{aidx} = vec_{cidx}_{aidx}_l.{method}(vec_{cidx}_{aidx}_r)"
                )
                body_lines.append(f"    masks.append(mask_{cidx}_{aidx})")
            else:
                raise ValueError("Unsupported atom types in codegen")

    # Now combine clauses: for each clause compute AND of its masks, then OR across clauses
    body_lines.append("")
    body_lines.append("    # Combine clause masks (AND within clause)")
    body_lines.append("    clause_masks = []")
    for cidx, clause in enumerate(clauses):
        if len(clause) == 1:
            mask_idx = sum(len(cl) for cl in clauses[:cidx])
            body_lines.append(f"    clause_masks.append(masks[{mask_idx}])")
        else:
            # combine sequentially using and_mask/or_mask
            start = sum(len(cl) for cl in clauses[:cidx])
            cur = f"masks[{start}]"
            for k in range(1, len(clause)):
                nxt = f"masks[{start + k}]"
                body_lines.append(f"    # combine {cur} AND {nxt}")
                body_lines.append(f"    {cur} = BoolMask({cur}).and_vector({nxt})")
            body_lines.append(f"    clause_masks.append({cur})")

    body_lines.append("")
    body_lines.append("    # OR across clause masks")
    body_lines.append("    if not clause_masks:")
    body_lines.append("        return BoolMask(bytes())")
    body_lines.append("    res = clause_masks[0]")
    body_lines.append("    for cm in clause_masks[1:]:")
    body_lines.append("        res = res.or_vector(cm)")
    body_lines.append("    return res")

    return "\n".join(header) + "\n\n" + "\n".join(body_lines)
