"""
Database CLI Commands

Commands for database management:
- reroute db init: Initialize migrations
- reroute db migrate: Create migration
- reroute db upgrade: Apply migrations
- reroute db downgrade: Rollback migrations
"""

import click
import sys
from pathlib import Path


@click.group()
def db():
    """Database management commands"""
    pass


@db.command()
def init():
    """
    Initialize database migrations

    Creates migrations directory and configuration files.
    """
    click.echo("\n🗄️  Initializing database migrations...")

    migrations_dir = Path("migrations")
    if migrations_dir.exists():
        click.secho("\n⚠️  Migrations directory already exists!", fg='yellow')
        return

    try:
        # Create migrations directory structure
        migrations_dir.mkdir(exist_ok=True)
        (migrations_dir / "versions").mkdir(exist_ok=True)

        # Create alembic.ini
        alembic_ini_content = """# Alembic Migration Configuration
# Generated by REROUTE CLI

[alembic]
script_location = migrations
prepend_sys_path = .
version_path_separator = os

[loggers]
keys = root,sqlalchemy,alembic

[handlers]
keys = console

[formatters]
keys = generic

[logger_root]
level = WARN
handlers = console

[logger_sqlalchemy]
level = WARN
handlers =
qualname = sqlalchemy.engine

[logger_alembic]
level = INFO
handlers =
qualname = alembic

[handler_console]
class = StreamHandler
args = (sys.stderr,)
level = NOTSET
formatter = generic

[formatter_generic]
format = %(levelname)-5.5s [%(name)s] %(message)s
datefmt = %H:%M:%S
"""
        Path("alembic.ini").write_text(alembic_ini_content)

        # Create env.py
        env_py_content = """from logging.config import fileConfig
from sqlalchemy import engine_from_config, pool
from alembic import context
import os
import sys

# Add project root to path
sys.path.insert(0, os.path.dirname(os.path.dirname(__file__)))

# Import your models here
from reroute.db.models import Base
# from app.models import *  # Import all your models

# Alembic Config object
config = context.config

# Interpret the config file for Python logging
if config.config_file_name is not None:
    fileConfig(config.config_file_name)

# Set target metadata
target_metadata = Base.metadata

# Get database URL from environment
database_url = os.getenv('REROUTE_DATABASE_URL', 'sqlite:///./app.db')
config.set_main_option('sqlalchemy.url', database_url)


def run_migrations_online():
    connectable = engine_from_config(
        config.get_section(config.config_ini_section),
        prefix='sqlalchemy.',
        poolclass=pool.NullPool,
    )

    with connectable.connect() as connection:
        context.configure(connection=connection, target_metadata=target_metadata)

        with context.begin_transaction():
            context.run_migrations()


if context.is_offline_mode():
    raise RuntimeError("Offline mode not supported")
else:
    run_migrations_online()
"""
        (migrations_dir / "env.py").write_text(env_py_content)

        click.secho("\n✅ Migrations initialized successfully!", fg='green')
        click.echo("\n📁 Created:")
        click.echo("  - migrations/")
        click.echo("  - migrations/versions/")
        click.echo("  - migrations/env.py")
        click.echo("  - alembic.ini")

        click.echo("\n💡 Next steps:")
        click.secho("  reroute db migrate -m 'Initial migration'", fg='cyan')
        click.secho("  reroute db upgrade", fg='cyan')
        click.echo()

    except Exception as e:
        click.secho(f"\n❌ Error: {e}", fg='red')
        sys.exit(1)


@db.command()
@click.option('-m', '--message', required=True, help='Migration message')
def migrate(message):
    """
    Create a new migration

    Example:
        reroute db migrate -m "Add users table"
    """
    click.echo(f"\n📝 Creating migration: {message}")

    try:
        import subprocess

        # Check if alembic is installed
        try:
            import alembic
        except ImportError:
            click.secho("\n❌ Alembic not installed!", fg='red')
            click.echo("Install with: pip install alembic")
            sys.exit(1)

        # Run alembic revision
        result = subprocess.run(
            ['alembic', 'revision', '--autogenerate', '-m', message],
            capture_output=True,
            text=True
        )

        if result.returncode == 0:
            click.secho("\n✅ Migration created successfully!", fg='green')
            click.echo(result.stdout)
        else:
            click.secho(f"\n❌ Error creating migration:", fg='red')
            click.echo(result.stderr)
            sys.exit(1)

    except Exception as e:
        click.secho(f"\n❌ Error: {e}", fg='red')
        sys.exit(1)


@db.command()
def upgrade():
    """
    Apply all pending migrations

    Applies migrations to the database.
    """
    click.echo("\n⬆️  Applying migrations...")

    try:
        import subprocess

        result = subprocess.run(
            ['alembic', 'upgrade', 'head'],
            capture_output=True,
            text=True
        )

        if result.returncode == 0:
            click.secho("\n✅ Migrations applied successfully!", fg='green')
            click.echo(result.stdout)
        else:
            click.secho(f"\n❌ Error applying migrations:", fg='red')
            click.echo(result.stderr)
            sys.exit(1)

    except Exception as e:
        click.secho(f"\n❌ Error: {e}", fg='red')
        sys.exit(1)


@db.command()
@click.option('--steps', default=1, help='Number of migrations to rollback')
def downgrade(steps):
    """
    Rollback migrations

    Example:
        reroute db downgrade --steps 1
    """
    click.echo(f"\n⬇️  Rolling back {steps} migration(s)...")

    try:
        import subprocess

        result = subprocess.run(
            ['alembic', 'downgrade', f'-{steps}'],
            capture_output=True,
            text=True
        )

        if result.returncode == 0:
            click.secho("\n✅ Rollback completed!", fg='green')
            click.echo(result.stdout)
        else:
            click.secho(f"\n❌ Error during rollback:", fg='red')
            click.echo(result.stderr)
            sys.exit(1)

    except Exception as e:
        click.secho(f"\n❌ Error: {e}", fg='red')
        sys.exit(1)


@db.command()
def current():
    """Show current migration version"""
    try:
        import subprocess

        result = subprocess.run(
            ['alembic', 'current'],
            capture_output=True,
            text=True
        )

        if result.returncode == 0:
            click.echo("\n📍 Current migration:")
            click.echo(result.stdout)
        else:
            click.secho(f"\n❌ Error:", fg='red')
            click.echo(result.stderr)

    except Exception as e:
        click.secho(f"\n❌ Error: {e}", fg='red')


@db.command()
def history():
    """Show migration history"""
    try:
        import subprocess

        result = subprocess.run(
            ['alembic', 'history'],
            capture_output=True,
            text=True
        )

        if result.returncode == 0:
            click.echo("\n📜 Migration history:")
            click.echo(result.stdout)
        else:
            click.secho(f"\n❌ Error:", fg='red')
            click.echo(result.stderr)

    except Exception as e:
        click.secho(f"\n❌ Error: {e}", fg='red')
