"""
Streamline plotting for 2D vector fields.

"""
from __future__ import (absolute_import, division, print_function,
                        unicode_literals)

import numpy as np
import matplotlib
import matplotlib.cm as cm
import matplotlib.colors as mcolors
import matplotlib.collections as mcollections
import matplotlib.lines as mlines
import matplotlib.patches as patches
import matplotlib.animation as animation

class TerminateTrajectory(Exception):
    pass




def velovect(axes, x, y, u, v, linewidth=None, color=None,
               cmap=None, norm=None, arrowsize=1, arrowstyle='-|>',
               transform=None, zorder=None, start_points=None,
               scale=1.0, grains=15, highlighted_regions=None, alpha=None):
    """Draws streamlines of a vector flow.

    *x*, *y* : 1d arrays
        an *evenly spaced* grid.
    *u*, *v* : 2d arrays
        x and y-velocities. Number of rows should match length of y, and
        the number of columns should match x.
    *density* : float or 2-tuple
        Controls the closeness of streamlines. When `density = 1`, the domain
        is divided into a 30x30 grid---*density* linearly scales this grid.
        Each cell in the grid can have, at most, one traversing streamline.
        For different densities in each direction, use [density_x, density_y].
    *linewidth* : numeric or 2d array
        vary linewidth when given a 2d array with the same shape as velocities.
    *color* : matplotlib color code, or 2d array
        Streamline color. When given an array with the same shape as
        velocities, *color* values are converted to colors using *cmap*.
    *cmap* : :class:`~matplotlib.colors.Colormap`
        Colormap used to plot streamlines and arrows. Only necessary when using
        an array input for *color*.
    *norm* : :class:`~matplotlib.colors.Normalize`
        Normalize object used to scale luminance data to 0, 1. If None, stretch
        (min, max) to (0, 1). Only necessary when *color* is an array.
    *arrowsize* : float
        Factor scale arrow size.
    *arrowstyle* : str
        Arrow style specification.
        See :class:`~matplotlib.patches.FancyArrowPatch`.
    *minlength* : float
        Minimum length of streamline in axes coordinates.
    *start_points*: Nx2 array
        Coordinates of starting points for the streamlines.
        In data coordinates, the same as the ``x`` and ``y`` arrays.
    *zorder* : int
        any number
    *scale* : float
        Maximum length of streamline in axes coordinates.
    *alpha* : float or 2d array
        Opacity of streamlines and arrows. When given a 2d array, must have same
        shape as velocities. Values should be between 0 and 1.

    Returns:

        *stream_container* : StreamplotSet
            Container object with attributes

                - lines: `matplotlib.collections.LineCollection` of streamlines

                - arrows: collection of `matplotlib.patches.FancyArrowPatch`
                  objects representing arrows half-way along stream
                  lines.

            This container will probably change in the future to allow changes
            to the colormap, alpha, etc. for both lines and arrows, but these
            changes should be backward compatible.

    """
    grid = Grid(x, y)
    mask = StreamMask(10)
    dmap = DomainMap(grid, mask)

    if zorder is None:
        zorder = mlines.Line2D.zorder

    # default to data coordinates
    if transform is None:
        transform = axes.transData

    if color is None:
        color = axes._get_lines.get_next_color()

    if linewidth is None:
        linewidth = matplotlib.rcParams['lines.linewidth']

    line_kw = {}
    arrow_kw = dict(arrowstyle=arrowstyle, mutation_scale=10 * arrowsize)

    use_multicolor_lines = isinstance(color, np.ndarray)
    if use_multicolor_lines:
        if color.shape != grid.shape:
            raise ValueError(
                "If 'color' is given, must have the shape of 'Grid(x,y)'")
        line_colors = []
        color = np.ma.masked_invalid(color)
    else:
        line_kw['color'] = color
        arrow_kw['color'] = color
        
    if isinstance(linewidth, np.ndarray):
        if linewidth.shape != grid.shape:
            raise ValueError(
                "If 'linewidth' is given, must have the shape of 'Grid(x,y)'")
        line_kw['linewidth'] = []
    else:
        line_kw['linewidth'] = linewidth
        arrow_kw['linewidth'] = linewidth

    if isinstance(alpha, np.ndarray):
        if alpha.shape != grid.shape:
            raise ValueError(
                "If 'alpha' is given as array, must have the shape of 'Grid(x,y)'")
        line_kw['alpha'] = []
    else:
        line_kw['alpha'] = alpha
        arrow_kw['alpha'] = alpha

    line_kw['zorder'] = zorder
    arrow_kw['zorder'] = zorder

    ## Sanity checks.
    if u.shape != grid.shape or v.shape != grid.shape:
        raise ValueError("'u' and 'v' must be of shape 'Grid(x,y)'")

    u = np.ma.masked_invalid(u)
    v = np.ma.masked_invalid(v)
    magnitude = np.sqrt(u**2 + v**2)
    magnitude/=np.max(magnitude)
	
    resolution = scale/grains
    minlength = 1*resolution
    integrate = get_integrator(u, v, dmap, minlength, resolution, magnitude)

    trajectories = []
    edges = []
    
    if start_points is None:
        start_points=_gen_starting_points(x,y,grains)
    
    sp2 = np.asanyarray(start_points, dtype=float).copy()

    # Check if start_points are outside the data boundaries
    for xs, ys in sp2:
        if not (grid.x_origin <= xs <= grid.x_origin + grid.width
                and grid.y_origin <= ys <= grid.y_origin + grid.height):
            raise ValueError("Starting point ({}, {}) outside of data "
                             "boundaries".format(xs, ys))

    # Convert start_points from data to array coords
    # Shift the seed points from the bottom left of the data so that
    # data2grid works properly.
    sp2[:, 0] -= grid.x_origin
    sp2[:, 1] -= grid.y_origin

    for xs, ys in sp2:
        xg, yg = dmap.data2grid(xs, ys)
        t = integrate(xg, yg)
        if t is not None:
            trajectories.append(t[0])
            edges.append(t[1])

    if use_multicolor_lines:
        if norm is None:
            norm = mcolors.Normalize(color.min(), color.max())
        if cmap is None:
            cmap = cm.get_cmap(matplotlib.rcParams['image.cmap'])
        else:
            cmap = cm.get_cmap(cmap)

    streamlines = []
    arrows = []
    
    for t, edge in zip(trajectories, edges):
        tgx = np.array(t[0])
        tgy = np.array(t[1])
        
		
        # Rescale from grid-coordinates to data-coordinates.
        tx, ty = dmap.grid2data(*np.array(t))
        tx += grid.x_origin
        ty += grid.y_origin

        
        points = np.transpose([tx, ty]).reshape(-1, 1, 2)
        streamlines.extend(np.hstack([points[:-1], points[1:]]))

        # Add arrows half way along each trajectory.
        s = np.cumsum(np.sqrt(np.diff(tx) ** 2 + np.diff(ty) ** 2))
        n = np.searchsorted(s, s[-1])
        arrow_tail = (tx[n], ty[n])
        arrow_head = (np.mean(tx[n:n + 2]), np.mean(ty[n:n + 2]))

        if isinstance(linewidth, np.ndarray):
            line_widths = interpgrid(linewidth, tgx, tgy)[:-1]
            line_kw['linewidth'].extend(line_widths)
            arrow_kw['linewidth'] = line_widths[n]

        if use_multicolor_lines:
            color_values = interpgrid(color, tgx, tgy)[:-1]
            line_colors.append(color_values)
            arrow_kw['color'] = cmap(norm(color_values[n]))
        
        if isinstance(alpha, np.ndarray):
            alpha_values = interpgrid(alpha, tgx, tgy)[:-1]
            line_kw['alpha'].extend(alpha_values)
            
            # Use maximum alpha value for the entire arrow to avoid partial highlighting
            arrow_kw['alpha'] = np.max(alpha_values)
        
        if not edge:
            p = patches.FancyArrowPatch(
                arrow_tail, arrow_head,
                transform=transform, **arrow_kw)
        else:
            continue
        
        ds = np.sqrt((arrow_tail[0]-arrow_head[0])**2+(arrow_tail[1]-arrow_head[1])**2)
        
        if ds<1e-15: continue #remove vanishingly short arrows that cause Patch to fail
        
        axes.add_patch(p)
        arrows.append(p)
        
        
    lc = mcollections.LineCollection(
        streamlines, transform=transform, **line_kw)
    lc.sticky_edges.x[:] = [grid.x_origin, grid.x_origin + grid.width]
    lc.sticky_edges.y[:] = [grid.y_origin, grid.y_origin + grid.height]
    if use_multicolor_lines:
        lc.set_array(np.ma.hstack(line_colors))
        lc.set_cmap(cmap)
        lc.set_norm(norm)
    axes.add_collection(lc)
    axes.autoscale_view()
    

    ac = matplotlib.collections.PatchCollection(arrows)
    stream_container = StreamplotSet(lc, ac)
    return stream_container

	

class StreamplotSet(object):

    def __init__(self, lines, arrows, **kwargs):
        self.lines = lines
        self.arrows = arrows


# Coordinate definitions
# ========================

class DomainMap(object):
    """Map representing different coordinate systems.

    Coordinate definitions:

    * axes-coordinates goes from 0 to 1 in the domain.
    * data-coordinates are specified by the input x-y coordinates.
    * grid-coordinates goes from 0 to N and 0 to M for an N x M grid,
      where N and M match the shape of the input data.
    * mask-coordinates goes from 0 to N and 0 to M for an N x M mask,
      where N and M are user-specified to control the density of streamlines.

    This class also has methods for adding trajectories to the StreamMask.
    Before adding a trajectory, run `start_trajectory` to keep track of regions
    crossed by a given trajectory. Later, if you decide the trajectory is bad
    (e.g., if the trajectory is very short) just call `undo_trajectory`.
    """

    def __init__(self, grid, mask):
        self.grid = grid
        self.mask = mask
        # Constants for conversion between grid- and mask-coordinates
        self.x_grid2mask = (mask.nx - 1) / grid.nx
        self.y_grid2mask = (mask.ny - 1) / grid.ny

        self.x_mask2grid = 1. / self.x_grid2mask
        self.y_mask2grid = 1. / self.y_grid2mask

        self.x_data2grid = 1. / grid.dx
        self.y_data2grid = 1. / grid.dy

    def grid2mask(self, xi, yi):
        """Return nearest space in mask-coords from given grid-coords."""
        return (int((xi * self.x_grid2mask) + 0.5),
                int((yi * self.y_grid2mask) + 0.5))

    def mask2grid(self, xm, ym):
        return xm * self.x_mask2grid, ym * self.y_mask2grid

    def data2grid(self, xd, yd):
        return xd * self.x_data2grid, yd * self.y_data2grid

    def grid2data(self, xg, yg):
        return xg / self.x_data2grid, yg / self.y_data2grid

    def start_trajectory(self, xg, yg):
        xm, ym = self.grid2mask(xg, yg)
        self.mask._start_trajectory(xm, ym)

    def reset_start_point(self, xg, yg):
        xm, ym = self.grid2mask(xg, yg)
        self.mask._current_xy = (xm, ym)

    def update_trajectory(self, xg, yg):
        
        xm, ym = self.grid2mask(xg, yg)
        #self.mask._update_trajectory(xm, ym)

    def undo_trajectory(self):
        self.mask._undo_trajectory()
        


class Grid(object):
    """Grid of data."""
    def __init__(self, x, y):

        if x.ndim == 1:
            pass
        elif x.ndim == 2:
            x_row = x[0, :]
            if not np.allclose(x_row, x):
                raise ValueError("The rows of 'x' must be equal")
            x = x_row
        else:
            raise ValueError("'x' can have at maximum 2 dimensions")

        if y.ndim == 1:
            pass
        elif y.ndim == 2:
            y_col = y[:, 0]
            if not np.allclose(y_col, y.T):
                raise ValueError("The columns of 'y' must be equal")
            y = y_col
        else:
            raise ValueError("'y' can have at maximum 2 dimensions")

        self.nx = len(x)
        self.ny = len(y)

        self.dx = x[1] - x[0]
        self.dy = y[1] - y[0]

        self.x_origin = x[0]
        self.y_origin = y[0]

        self.width = x[-1] - x[0]
        self.height = y[-1] - y[0]

    @property
    def shape(self):
        return self.ny, self.nx

    def within_grid(self, xi, yi):
        """Return True if point is a valid index of grid."""
        # Note that xi/yi can be floats; so, for example, we can't simply check
        # `xi < self.nx` since `xi` can be `self.nx - 1 < xi < self.nx`
        return xi >= 0 and xi <= self.nx - 1 and yi >= 0 and yi <= self.ny - 1


class StreamMask(object):
    """Mask to keep track of discrete regions crossed by streamlines.

    The resolution of this grid determines the approximate spacing between
    trajectories. Streamlines are only allowed to pass through zeroed cells:
    When a streamline enters a cell, that cell is set to 1, and no new
    streamlines are allowed to enter.
    """

    def __init__(self, density):
        if np.isscalar(density):
            if density <= 0:
                raise ValueError("If a scalar, 'density' must be positive")
            self.nx = self.ny = int(30 * density)
        else:
            if len(density) != 2:
                raise ValueError("'density' can have at maximum 2 dimensions")
            self.nx = int(30 * density[0])
            self.ny = int(30 * density[1])
        self._mask = np.zeros((self.ny, self.nx))
        self.shape = self._mask.shape

        self._current_xy = None

    def __getitem__(self, *args):
        return self._mask.__getitem__(*args)

    def _start_trajectory(self, xm, ym):
        """Start recording streamline trajectory"""
        self._traj = []
        self._update_trajectory(xm, ym)

    def _undo_trajectory(self):
        """Remove current trajectory from mask"""
        for t in self._traj:
            self._mask.__setitem__(t, 0)

    def _update_trajectory(self, xm, ym):
        """Update current trajectory position in mask.

        If the new position has already been filled, raise `InvalidIndexError`.
        """
        #if self._current_xy != (xm, ym):
        #    if self[ym, xm] == 0:
        self._traj.append((ym, xm))
        self._mask[ym, xm] = 1
        self._current_xy = (xm, ym)
        #    else:
        #        raise InvalidIndexError




# Integrator definitions
#========================

def get_integrator(u, v, dmap, minlength, resolution, magnitude):

    # rescale velocity onto grid-coordinates for integrations.
    u, v = dmap.data2grid(u, v)

    # speed (path length) will be in axes-coordinates
    u_ax = u / dmap.grid.nx
    v_ax = v / dmap.grid.ny
    speed = np.ma.sqrt(u_ax ** 2 + v_ax ** 2)

    def forward_time(xi, yi):
        ds_dt = interpgrid(speed, xi, yi)
        if ds_dt == 0:
            raise TerminateTrajectory()
        dt_ds = 1. / ds_dt
        ui = interpgrid(u, xi, yi)
        vi = interpgrid(v, xi, yi)
        return ui * dt_ds, vi * dt_ds


    def integrate(x0, y0):
        """Return x, y grid-coordinates of trajectory based on starting point.

        Integrate both forward and backward in time from starting point in
        grid coordinates.

        Integration is terminated when a trajectory reaches a domain boundary
        or when it crosses into an already occupied cell in the StreamMask. The
        resulting trajectory is None if it is shorter than `minlength`.
        """

        stotal, x_traj, y_traj = 0., [], []

        
        dmap.start_trajectory(x0, y0)

        dmap.reset_start_point(x0, y0)
        stotal, x_traj, y_traj, m_total, hit_edge = _integrate_rk12(x0, y0, dmap, forward_time, resolution, magnitude)

        
        if len(x_traj)>1:
            return (x_traj, y_traj), hit_edge
        else:  # reject short trajectories
            dmap.undo_trajectory()
            return None

    return integrate


def _integrate_rk12(x0, y0, dmap, f, resolution, magnitude):
    """2nd-order Runge-Kutta algorithm with adaptive step size.

    This method is also referred to as the improved Euler's method, or Heun's
    method. This method is favored over higher-order methods because:

    1. To get decent looking trajectories and to sample every mask cell
       on the trajectory we need a small timestep, so a lower order
       solver doesn't hurt us unless the data is *very* high resolution.
       In fact, for cases where the user inputs
       data smaller or of similar grid size to the mask grid, the higher
       order corrections are negligible because of the very fast linear
       interpolation used in `interpgrid`.

    2. For high resolution input data (i.e. beyond the mask
       resolution), we must reduce the timestep. Therefore, an adaptive
       timestep is more suited to the problem as this would be very hard
       to judge automatically otherwise.

    This integrator is about 1.5 - 2x as fast as both the RK4 and RK45
    solvers in most setups on my machine. I would recommend removing the
    other two to keep things simple.
    """
    # This error is below that needed to match the RK4 integrator. It
    # is set for visual reasons -- too low and corners start
    # appearing ugly and jagged. Can be tuned.
    maxerror = 0.003

    # This limit is important (for all integrators) to avoid the
    # trajectory skipping some mask cells. We could relax this
    # condition if we use the code which is commented out below to
    # increment the location gradually. However, due to the efficient
    # nature of the interpolation, this doesn't boost speed by much
    # for quite a bit of complexity.
    maxds = min(1. / dmap.mask.nx, 1. / dmap.mask.ny, 0.1)

    ds = maxds
    stotal = 0
    xi = x0
    yi = y0
    xf_traj = []
    yf_traj = []
    m_total = []
    hit_edge = False
    
    while dmap.grid.within_grid(xi, yi):
        xf_traj.append(xi)
        yf_traj.append(yi)
        m_total.append(interpgrid(magnitude, xi, yi))
        try:
            k1x, k1y = f(xi, yi)
            k2x, k2y = f(xi + ds * k1x,
                         yi + ds * k1y)
        except IndexError:
            # Out of the domain on one of the intermediate integration steps.
            # Take an Euler step to the boundary to improve neatness.
            ds, xf_traj, yf_traj = _euler_step(xf_traj, yf_traj, dmap, f)
            stotal += ds
            hit_edge = True
            break
        except TerminateTrajectory:
            break

        dx1 = ds * k1x
        dy1 = ds * k1y
        dx2 = ds * 0.5 * (k1x + k2x)
        dy2 = ds * 0.5 * (k1y + k2y)

        nx, ny = dmap.grid.shape
        # Error is normalized to the axes coordinates
        error = np.sqrt(((dx2 - dx1) / nx) ** 2 + ((dy2 - dy1) / ny) ** 2)

        # Only save step if within error tolerance
        if error < maxerror:
            xi += dx2
            yi += dy2
            
            dmap.update_trajectory(xi, yi)
            
            if not dmap.grid.within_grid(xi, yi):
                hit_edge=True
            
            if (stotal + ds) > resolution*np.mean(m_total):
                break
            stotal += ds

        # recalculate stepsize based on step error
        if error == 0:
            ds = maxds
        else:
            ds = min(maxds, 0.85 * ds * (maxerror / error) ** 0.5)

    return stotal, xf_traj, yf_traj, m_total, hit_edge


def _euler_step(xf_traj, yf_traj, dmap, f):
    """Simple Euler integration step that extends streamline to boundary."""
    ny, nx = dmap.grid.shape
    xi = xf_traj[-1]
    yi = yf_traj[-1]
    cx, cy = f(xi, yi)
    if cx == 0:
        dsx = np.inf
    elif cx < 0:
        dsx = xi / -cx
    else:
        dsx = (nx - 1 - xi) / cx
    if cy == 0:
        dsy = np.inf
    elif cy < 0:
        dsy = yi / -cy
    else:
        dsy = (ny - 1 - yi) / cy
    ds = min(dsx, dsy)
    xf_traj.append(xi + cx * ds)
    yf_traj.append(yi + cy * ds)
    return ds, xf_traj, yf_traj


# Utility functions
# ========================

def interpgrid(a, xi, yi):
    """Fast 2D, linear interpolation on an integer grid"""

    Ny, Nx = np.shape(a)
    if isinstance(xi, np.ndarray):
        x = xi.astype(int)
        y = yi.astype(int)
        # Check that xn, yn don't exceed max index
        xn = np.clip(x + 1, 0, Nx - 1)
        yn = np.clip(y + 1, 0, Ny - 1)
    else:
        x = int(xi)
        y = int(yi)
        # conditional is faster than clipping for integers
        if x == (Nx - 2):
            xn = x
        else:
            xn = x + 1
        if y == (Ny - 2):
            yn = y
        else:
            yn = y + 1

    a00 = a[y, x]
    a01 = a[y, xn]
    a10 = a[yn, x]
    a11 = a[yn, xn]
    xt = xi - x
    yt = yi - y
    a0 = a00 * (1 - xt) + a01 * xt
    a1 = a10 * (1 - xt) + a11 * xt
    ai = a0 * (1 - yt) + a1 * yt

    if not isinstance(xi, np.ndarray):
        if np.ma.is_masked(ai):
            raise TerminateTrajectory

    return ai


def _gen_starting_points(x,y,grains):
    
    eps = np.finfo(np.float32).eps
    
    tmp_x =  np.linspace(x.min()+eps, x.max()-eps, grains)
    tmp_y =  np.linspace(y.min()+eps, y.max()-eps, grains)
    
    xs = np.tile(tmp_x, grains)
    ys = np.repeat(tmp_y, grains)

    seed_points = np.array([list(xs), list(ys)])
    
    return seed_points.T


def animate_velovect(axes, x, y, u, v, n_particles=100, frames=100, interval=50,
                    linewidth=None, color=None, cmap=None, norm=None, 
                    arrowsize=1, arrowstyle='-|>', transform=None, zorder=None,
                    scale=1.0, grains=15, highlighted_regions=None, alpha=None,
                    particle_color='r', particle_size=5, save_path=None, fps=30,
                    tail_length=10):
    """Animates a vector field with moving particles showing the flow.

    Parameters
    ----------
    axes : matplotlib.axes.Axes
        Axes object to draw the animation on.
    x, y : 1d arrays
        An evenly spaced grid.
    u, v : 2d arrays
        x and y-velocities. Number of rows should match length of y, and
        the number of columns should match x.
    n_particles : int, optional
        Number of particles to animate.
    frames : int, optional
        Number of frames in the animation.
    interval : int, optional
        Delay between frames in milliseconds.
    linewidth, color, cmap, norm, arrowsize, arrowstyle, transform, zorder,
    scale, grains, highlighted_regions, alpha : 
        Parameters passed to velovect.
    particle_color : str, optional
        Color of the particles.
    particle_size : float, optional
        Size of the particles.
    save_path : str, optional
        If provided, animation will be saved to this path.
    fps : int, optional
        Frames per second for saved animation.
    tail_length : int, optional
        Number of previous positions to show as a tail.

    Returns
    -------
    anim : matplotlib.animation.FuncAnimation
        Animation object.
    """
    # Create the static vector field using velovect
    stream_container = velovect(axes, x, y, u, v, linewidth, color, cmap, norm,
                              arrowsize, arrowstyle, transform, zorder, None,
                              scale, grains, highlighted_regions, alpha)
    
    # Create a grid for interpolation
    grid = Grid(x, y)
    
    # Create random starting points for particles
    rand_indices = np.random.randint(0, len(x) * len(y), size=n_particles)
    particles_x = np.array([x[i % len(x)] for i in rand_indices])
    particles_y = np.array([y[i // len(x)] for i in rand_indices])
    
    # Initialize particle positions and history
    # For each particle, keep track of multiple trail segments
    particle_trails = [[] for _ in range(n_particles)]
    for i in range(n_particles):
        particle_trails[i].append([(particles_x[i], particles_y[i])])
    
    # Create particle scatter plot
    particles = axes.scatter(particles_x, particles_y, color=particle_color, 
                            s=particle_size, zorder=zorder+1 if zorder else 3)
    
    # Create particle tails (empty initially)
    if tail_length > 0:
        tails = [axes.plot([], [], color=particle_color, alpha=0.5, 
                          linewidth=particle_size/2)[0] for _ in range(n_particles)]
    else:
        tails = None
    
    # Normalize velocity for consistent particle movement
    magnitude = np.sqrt(u**2 + v**2)
    max_mag = np.max(magnitude)
    if max_mag > 0:  # Avoid division by zero
        u_norm = u / max_mag * scale
        v_norm = v / max_mag * scale
    else:
        u_norm = u * 0
        v_norm = v * 0
    
    # Function to update particle positions
    def update(frame):
        current_x = np.array([particle_trails[i][-1][-1][0] for i in range(n_particles)])
        current_y = np.array([particle_trails[i][-1][-1][1] for i in range(n_particles)])
        next_x = current_x.copy()
        next_y = current_y.copy()
        
        # Move particles according to the velocity field
        for i in range(n_particles):
            # Find grid indices for interpolation
            x_idx = np.clip(np.searchsorted(x, current_x[i]) - 1, 0, len(x) - 2)
            y_idx = np.clip(np.searchsorted(y, current_y[i]) - 1, 0, len(y) - 2)
            
            # Calculate interpolation weights
            x_weight = (current_x[i] - x[x_idx]) / (x[x_idx + 1] - x[x_idx])
            y_weight = (current_y[i] - y[y_idx]) / (y[y_idx + 1] - y[y_idx])
            
            # Bilinear interpolation of velocity
            u_interp = (1 - x_weight) * (1 - y_weight) * u_norm[y_idx, x_idx] + \
                      x_weight * (1 - y_weight) * u_norm[y_idx, x_idx + 1] + \
                      (1 - x_weight) * y_weight * u_norm[y_idx + 1, x_idx] + \
                      x_weight * y_weight * u_norm[y_idx + 1, x_idx + 1]
                      
            v_interp = (1 - x_weight) * (1 - y_weight) * v_norm[y_idx, x_idx] + \
                      x_weight * (1 - y_weight) * v_norm[y_idx, x_idx + 1] + \
                      (1 - x_weight) * y_weight * v_norm[y_idx + 1, x_idx] + \
                      x_weight * y_weight * v_norm[y_idx + 1, x_idx + 1]
            
            # Update position
            next_x[i] += u_interp
            next_y[i] += v_interp
            
            # Check if the particle is still in bounds
            in_bounds = (next_x[i] >= x.min() and next_x[i] <= x.max() and 
                         next_y[i] >= y.min() and next_y[i] <= y.max())
            
            if in_bounds:
                # Add new position to the current trail
                particle_trails[i][-1].append((next_x[i], next_y[i]))
                
                # Keep only the last tail_length positions in the trail
                if len(particle_trails[i][-1]) > tail_length:
                    particle_trails[i][-1] = particle_trails[i][-1][-tail_length:]
            else:
                # Reset particle to a new random position
                next_x[i] = x[np.random.randint(0, len(x))]
                next_y[i] = y[np.random.randint(0, len(y))]
                
                # Start a new trail segment
                particle_trails[i].append([(next_x[i], next_y[i])])
                
                # Keep only the last few trail segments to limit memory usage
                if len(particle_trails[i]) > 5:
                    particle_trails[i] = particle_trails[i][-5:]
        
        # Update scatter plot
        particles.set_offsets(np.column_stack([next_x, next_y]))
        
        # Update tails
        if tails and tail_length > 0:
            for i in range(n_particles):
                # Only plot the current trail segment (not connecting across resets)
                if particle_trails[i][-1]:
                    xs, ys = zip(*particle_trails[i][-1])
                    tails[i].set_data(xs, ys)
                else:
                    tails[i].set_data([], [])
        
        return [particles] + (tails if tails else [])
    
    # Create animation
    anim = animation.FuncAnimation(axes.figure, update, frames=frames, 
                                  interval=interval, blit=True, 
                                  repeat=True)
    
    # Save animation if path is provided
    if save_path:
        anim.save(save_path, fps=fps)
    
    return anim



