# SPDX-FileCopyrightText: PhiBo DinoTools (2021)
# SPDX-License-Identifier: GPL-3.0-or-later

import re
from typing import Optional, Tuple

import click
import nagiosplugin

from ..cli import cli
from ..helper import logger
from ..resource import RouterOSCheckResource


class ToolPingCheck(RouterOSCheckResource):
    name = "PING"

    def __init__(self, cmd_options, check: nagiosplugin.Check, address):
        super().__init__(cmd_options=cmd_options, check=check)

        self._address = address
        self._max_packages = 1

    def probe(self):
        def strip_time(value) -> Tuple[Optional[int], Optional[str]]:
            m = re.compile(r"^(?P<time>[0-9]+)(?P<uom>.*)$").match(value)
            if m:
                return int(m.group("time")), m.group("uom")
            return None, None

        params = {"address": self._address, "count": self._max_packages}
        api = self._connect_api()

        logger.info("Call /ping command ...")
        call = api("/ping", **params)
        results = tuple(call)
        result = results[-1]

        yield nagiosplugin.Metric(
            name="packet_loss",
            value=result["packet-loss"],
            uom="%",
            min=0,
            max=100,
        )
        yield nagiosplugin.Metric(
            name="sent",
            value=result["sent"],
            min=0,
            max=self._max_packages,
        )
        yield nagiosplugin.Metric(
            name="received",
            value=result["received"],
            min=0,
            max=self._max_packages,
        )

        if result["received"] > 0:
            yield nagiosplugin.Metric(
                name="rtt_min",
                value=strip_time(result["min-rtt"])[0],
                min=0,
            )
            yield nagiosplugin.Metric(
                name="rtt_max",
                value=strip_time(result["max-rtt"])[0],
                min=0,
            )
            yield nagiosplugin.Metric(
                name="rtt_avg",
                value=strip_time(result["avg-rtt"])[0],
                min=0,
            )
            yield nagiosplugin.Metric(
                name="size",
                value=result["size"]
            )
            yield nagiosplugin.Metric(
                name="ttl",
                value=result["ttl"],
                min=0,
                max=255,
            )


@cli.command("tool.ping")
@click.option(
    "--address",
    required=True,
    help="Address of device to ping",
)
@click.option(
    "--packet-loss-warning",
    help="Warning threshold for packet loss",
)
@click.option(
    "--packet-loss-critical",
    help="Critical threshold for packet loss",
)
@click.option(
    "--ttl-warning",
    help="Warning threshold for the Time-To-Live (TTL) value",
)
@click.option(
    "--ttl-critical",
    help="Critical threshold for the Time-To-Live (TTL) value",
)
@click.pass_context
def tool_ping(ctx, address, packet_loss_warning, packet_loss_critical, ttl_warning, ttl_critical):
    """Execute a ping command on the device to check other devices"""
    check = nagiosplugin.Check()

    check.add(
        ToolPingCheck(
            cmd_options=ctx.obj,
            check=check,
            address=address
        ),
        nagiosplugin.ScalarContext(
            name="packet_loss",
            warning=packet_loss_warning,
            critical=packet_loss_critical
        ),
        nagiosplugin.ScalarContext(
            name="sent"
        ),
        nagiosplugin.ScalarContext(
            name="received"
        ),
        nagiosplugin.ScalarContext(
            name="rtt_avg"
        ),
        nagiosplugin.ScalarContext(
            name="rtt_min"
        ),
        nagiosplugin.ScalarContext(
            name="rtt_max"
        ),
        nagiosplugin.ScalarContext(
            name="size"
        ),
        nagiosplugin.ScalarContext(
            name="ttl",
            warning=ttl_warning,
            critical=ttl_critical
        ),
    )

    check.main(verbose=ctx.obj["verbose"])
