# 📧 EzMail

**Send and read emails easily — with attachments, inline images, HTML templates, and OAuth2 authentication.**

`ezmail` is a modern Python library that simplifies email automation.
It allows you to **send and receive emails** via SMTP and IMAP, supporting advanced features such as inline images, HTML content, file attachments, and secure authentication (TLS or OAuth2).

---

## 🚀 Features

### ✉️ Sending Emails (EzSender)

* Send emails to one or multiple recipients
* Support for both **HTML** and **plain text** bodies
* Add **inline images** directly into the message
* Attach any type of file (`PDF`, `CSV`, `XML`, etc.)
* Use **Jinja2 templates** for dynamic HTML emails
* Secure connection using **TLS/SSL**
* Optional hourly send rate limit

### 📥 Reading Emails (EzReader)

* Connect via **IMAP** securely using password or **OAuth2 token**
* List mailboxes (`INBOX`, `Sent`, `Drafts`, etc.)
* Filter by: `ALL`, `UNSEEN`, `FROM`, `SUBJECT`, `TEXT`, `SINCE`, `BEFORE`
* Retrieve **attachments in memory** (not saved automatically)
* Return results as structured `EzMail` objects

---

## 💻 Installation

```bash
pip install ezmail
```

No extra configuration needed — just use your SMTP and IMAP credentials.

---

## 🧠 Quick Overview

| Class      | Description                                                          |
| ---------- | -------------------------------------------------------------------- |
| `EzSender` | Composes and sends emails with HTML, inline images, and attachments. |
| `EzReader` | Reads and filters emails from any IMAP server.                       |
| `EzMail`   | Represents a single email (sender, subject, body, attachments).      |

---

## ✉️ Example — Sending Emails

```python
from ezmail import EzSender

smtp = {"server": "smtp.gmail.com", "port": 587}
sender = {"email": "me@gmail.com", "password": "app_password"}

ez = EzSender(smtp, sender)
ez.subject = "System Update Report"
ez.add_text("<h2>Hello!</h2><p>The latest system report is attached below.</p>")
ez.add_attachment("report.pdf")

ez.send(["client@example.com", "team@example.com"])
```

---

## 📬 Example — Reading Emails

```python
from ezmail import EzReader

imap = {"server": "imap.gmail.com", "port": 993}
account = {
    "email": "me@gmail.com",
    "auth_value": "oauth2_token_or_password",
    "auth_type": "password"  # or "oauth2"
}

reader = EzReader(imap, account)
reader.connect()

emails = reader.fetch_messages(status="UNSEEN", since="01-Oct-2025")
for mail in emails:
    print(mail.subject, "-", mail.sender)
    if mail.has_attachments():
        for a in mail.attachments:
            print("💎", a["filename"], len(a["data"]), "bytes")

reader.disconnect()
```

---

## 🯩 Advanced Example — Using Templates and Inline Images

```python
ez = EzSender(
    smtp={"server": "smtp.domain.com", "port": 587},
    sender={"email": "me@domain.com", "password": "mypassword"}
)

ez.subject = "Welcome to our platform!"
ez.use_template("templates/welcome.html", name="John", version="3.2.1")
ez.add_image("logo.png", width="150px", cid="logo_img")
ez.send("john@client.com")
```

---

## 🔐 Authentication Methods

| Method     | Description                                                                     |
| ---------- | ------------------------------------------------------------------------------- |
| `password` | Standard login using email and password (supports app-specific passwords).      |
| `oauth2`   | Secure OAuth2 token authentication — required for Gmail and Microsoft accounts. |

---

## 📦 Dependencies

* [Jinja2](https://pypi.org/project/Jinja2/) ≥ 3.0.0 (for HTML templates)
* Built-in modules only: `smtplib`, `imaplib`, `email`, `mimetypes`, `uuid`, etc.

---

## 🧮 Requirements

* Python ≥ 3.8
* Internet access (for SMTP/IMAP servers)

---

## 🧳 License

MIT © [Luiz Henrique Brunca](https://github.com/luizbrunca)

---

## 🌎 Other Languages

* 🇧🇷 **[Leia em Português (README.pt-BR.md)](https://github.com/LuizBrunca/ezmail/blob/main/README.pt-BR.md)**
