import { CfnResource, IAspect } from 'aws-cdk-lib';
import { IConstruct } from 'constructs';
import { INagSuppressionIgnore } from './ignore-suppression-conditions';
import { NagPackSuppression } from './models/nag-suppression';
import { INagLogger, NagReportFormat } from './nag-logger';
import { NagMessageLevel, NagRuleResult } from './nag-rules';
/**
 * Interface for creating a NagPack.
 */
export interface NagPackProps {
    /**
     * Whether or not to enable extended explanatory descriptions on warning, error, and logged ignore messages (default: false).
     */
    readonly verbose?: boolean;
    /**
     * Whether or not to log suppressed rule violations as informational messages (default: false).
     */
    readonly logIgnores?: boolean;
    /**
     * Whether or not to generate compliance reports for applied Stacks in the App's output directory (default: true).
     */
    readonly reports?: boolean;
    /**
     * If reports are enabled, the output formats of compliance reports in the App's output directory (default: only CSV).
     */
    readonly reportFormats?: NagReportFormat[];
    /**
     * Conditionally prevent rules from being suppressed (default: no user provided condition).
     */
    readonly suppressionIgnoreCondition?: INagSuppressionIgnore;
    /**
     * Additional NagLoggers for logging rule validation outputs.
     */
    readonly additionalLoggers?: INagLogger[];
}
/**
 * Interface for JSII interoperability for passing parameters and the Rule Callback to @applyRule method.
 */
export interface IApplyRule {
    /**
     * Override for the suffix of the Rule ID for this rule
     */
    ruleSuffixOverride?: string;
    /**
     * Why the rule was triggered.
     */
    info: string;
    /**
     * Why the rule exists.
     */
    explanation: string;
    /**
     * The annotations message level to apply to the rule if triggered.
     */
    level: NagMessageLevel;
    /**
     * A condition in which a suppression should be ignored.
     */
    ignoreSuppressionCondition?: INagSuppressionIgnore;
    /**
     * The CfnResource to check
     */
    node: CfnResource;
    /**
     * The callback to the rule.
     * @param node The CfnResource to check.
     */
    rule(node: CfnResource): NagRuleResult;
}
/**
 * Base class for all rule packs.
 */
export declare abstract class NagPack implements IAspect {
    protected loggers: INagLogger[];
    protected packName: string;
    protected userGlobalSuppressionIgnore?: INagSuppressionIgnore;
    protected packGlobalSuppressionIgnore?: INagSuppressionIgnore;
    constructor(props?: NagPackProps);
    get readPackName(): string;
    /**
     * All aspects can visit an IConstruct.
     */
    abstract visit(node: IConstruct): void;
    /**
     * Create a rule to be used in the NagPack.
     * @param params The @IApplyRule interface with rule details.
     */
    protected applyRule(params: IApplyRule): void;
    /**
     * Check whether a specific rule should be ignored.
     * @param suppressions The suppressions listed in the cdk-nag metadata.
     * @param ruleId The id of the rule to ignore.
     * @param resource The resource being evaluated.
     * @param findingId The id of the finding that is being checked.
     * @param validationFailure Whether the rule is being checked due to a validation failure.
     * @returns The reason the rule was ignored, or an empty string.
     */
    protected ignoreRule(suppressions: NagPackSuppression[], ruleId: string, findingId: string, resource: CfnResource, level: NagMessageLevel, ignoreSuppressionCondition?: INagSuppressionIgnore, validationFailure?: boolean): string;
    private isNonCompliant;
    private asFindings;
}
