"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NagSuppressions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_suppression_helper_1 = require("./utils/nag-suppression-helper");
/**
 * Helper class with methods to add cdk-nag suppressions to cdk resources
 */
class NagSuppressions {
    /**
     * Apply cdk-nag suppressions to a Stack and optionally nested stacks
     * @param stack The Stack to apply the suppression to
     * @param suppressions A list of suppressions to apply to the stack
     * @param applyToNestedStacks Apply the suppressions to children stacks (default:false)
     */
    static addStackSuppressions(stack, suppressions, applyToNestedStacks = false) {
        const stacks = applyToNestedStacks
            ? stack.node.findAll().filter((x) => x instanceof aws_cdk_lib_1.Stack)
            : [stack];
        stacks.forEach((s) => {
            nag_suppression_helper_1.NagSuppressionHelper.assertSuppressionsAreValid(s.node.id, suppressions);
            let metadata = s.templateOptions.metadata?.cdk_nag ?? {};
            metadata = nag_suppression_helper_1.NagSuppressionHelper.addRulesToMetadata(metadata, suppressions);
            if (!s.templateOptions.metadata) {
                s.templateOptions.metadata = {};
            }
            s.templateOptions.metadata.cdk_nag = metadata;
        });
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children
     * @param construct The IConstruct(s) to apply the suppression to
     * @param suppressions A list of suppressions to apply to the resource
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false)
     */
    static addResourceSuppressions(construct, suppressions, applyToChildren = false) {
        const constructArray = Array.isArray(construct)
            ? construct
            : [construct];
        if (constructArray.length === 0) {
            throw new Error(`Construct cannot be an empty array.`);
        }
        constructArray.forEach((c) => {
            nag_suppression_helper_1.NagSuppressionHelper.assertSuppressionsAreValid(c.node.id, suppressions);
            const constructs = applyToChildren ? c.node.findAll() : [c];
            for (const child of constructs) {
                const possibleL1 = child.node.defaultChild
                    ? child.node.defaultChild
                    : child;
                if (possibleL1 instanceof aws_cdk_lib_1.CfnResource) {
                    const resource = possibleL1;
                    let metadata = resource.getMetadata('cdk_nag');
                    metadata = nag_suppression_helper_1.NagSuppressionHelper.addRulesToMetadata(metadata, suppressions);
                    resource.addMetadata('cdk_nag', metadata);
                }
            }
        });
    }
    /**
     * Add cdk-nag suppressions to a CfnResource and optionally its children via its path
     * @param stack The Stack the construct belongs to
     * @param path The path(s) to the construct in the provided stack
     * @param suppressions A list of suppressions to apply to the resource
     * @param applyToChildren Apply the suppressions to children CfnResources  (default:false)
     */
    static addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren = false) {
        const pathArray = Array.isArray(path) ? path : [path];
        if (pathArray.length === 0) {
            throw new Error(`Path cannot be an empty array.`);
        }
        pathArray.forEach((p) => {
            let added = false;
            for (const child of stack.node.findAll()) {
                const fixedPath = p.replace(/^\//, '');
                if (child.node.path === fixedPath ||
                    child.node.path + '/Resource' === fixedPath) {
                    NagSuppressions.addResourceSuppressions(child, suppressions, applyToChildren);
                    added = true;
                }
            }
            if (!added) {
                throw new Error(`Suppression path "${p}" did not match any resource. This can occur when a resource does not exist or if a suppression is applied before a resource is created.`);
            }
        });
    }
}
exports.NagSuppressions = NagSuppressions;
_a = JSII_RTTI_SYMBOL_1;
NagSuppressions[_a] = { fqn: "cdk-nag.NagSuppressions", version: "2.37.30" };
//# sourceMappingURL=data:application/json;base64,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