"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const nag_rules_1 = require("../../nag-rules");
/**
 * EKS Cluster Kubernetes API server endpoints have public access disabled
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_eks_1.CfnCluster) {
        const resourcesVpcConfig = aws_cdk_lib_1.Stack.of(node).resolve(node.resourcesVpcConfig);
        const endpointPublicAccess = nag_rules_1.NagRules.resolveIfPrimitive(node, resourcesVpcConfig.endpointPublicAccess);
        if (endpointPublicAccess !== false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node.cfnResourceType === 'Custom::AWSCDK-EKS-Cluster') {
        // The CDK uses a Custom Resource with AWS SDK calls to create EKS Clusters
        const props = aws_cdk_lib_1.Stack.of(node).resolve(node._cfnProperties);
        const endpointPublicAccess = nag_rules_1.NagRules.resolveIfPrimitive(node, props?.Config?.resourcesVpcConfig?.endpointPublicAccess);
        if (endpointPublicAccess !== false) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,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