"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_rds_1 = require("aws-cdk-lib/aws-rds");
const nag_rules_1 = require("../../nag-rules");
/**
 * Non-Aurora RDS DB instances and Aurora clusters are configured to export all possible log types to CloudWatch
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_rds_1.CfnDBInstance) {
        const dbType = String(nag_rules_1.NagRules.resolveIfPrimitive(node, node.engine));
        const exports = aws_cdk_lib_1.Stack.of(node).resolve(node.enableCloudwatchLogsExports) ?? [];
        const needed = [];
        if (!dbType.includes('aurora')) {
            if (dbType.includes('mariadb') || dbType.includes('mysql')) {
                needed.push('audit', 'error', 'general', 'slowquery');
            }
            else if (dbType.includes('postgres')) {
                needed.push('postgresql', 'upgrade');
            }
            else if (dbType.includes('oracle')) {
                needed.push('audit', 'alert', 'listener', 'oemagent', 'trace');
            }
            else if (dbType.includes('sqlserver')) {
                needed.push('agent', 'error');
            }
            const findings = needed
                .filter((log) => !exports.includes(log))
                .map((log) => `LogExport::${log}`);
            return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
        }
        else {
            return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
        }
    }
    else if (node instanceof aws_rds_1.CfnDBCluster) {
        const engine = nag_rules_1.NagRules.resolveIfPrimitive(node, node.engine).toLowerCase();
        const needed = [];
        const exports = aws_cdk_lib_1.Stack.of(node).resolve(node.enableCloudwatchLogsExports) ?? [];
        if (engine.toLowerCase() === 'aurora' ||
            engine.toLowerCase() === 'aurora-mysql') {
            needed.push('audit', 'error', 'general', 'slowquery');
        }
        else if (engine.toLowerCase() === 'aurora-postgresql') {
            needed.push('postgresql');
        }
        const findings = needed
            .filter((log) => !exports.includes(log))
            .map((log) => `LogExport::${log}`);
        return findings.length ? findings : nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
//# sourceMappingURL=data:application/json;base64,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