"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.flattenCfnReference = void 0;
/**
 * Turns a CloudFormation reference object into a flat string for easy suppression
 * @param node The reference object
 * @returns Flattened string
 */
const flattenCfnReference = (reference) => {
    const visit = (node) => {
        if (typeof node === 'undefined') {
            return '';
        }
        if (typeof node === 'string') {
            // Replace the template syntax that Fn::Sub uses with round brackets for easier suppression
            return node.replace(/\${/g, '<').replace(/}/g, '>');
        }
        if (isCfnFnJoin(node)) {
            const [delimiter, items] = node['Fn::Join'];
            return items.map(visit).join(delimiter);
        }
        if (isCfnFnSub(node)) {
            return visit(node['Fn::Sub']);
        }
        if (isCfnFnGetAtt(node)) {
            const [resource, attribute] = node['Fn::GetAtt'];
            return `<${visit(resource)}.${visit(attribute)}>`;
        }
        if (isCfnFnImportValue(node)) {
            return visit(node['Fn::ImportValue']);
        }
        if (isCfnRef(node)) {
            return `<${visit(node.Ref)}>`;
        }
        // fallback
        return JSON.stringify(node);
    };
    return visit(reference);
};
exports.flattenCfnReference = flattenCfnReference;
// Checks if an object is a certain type by testing for the existence of a field
const checkType = (field) => (node) => !!node[field];
const isCfnFnJoin = checkType('Fn::Join');
const isCfnFnSub = checkType('Fn::Sub');
const isCfnFnGetAtt = checkType('Fn::GetAtt');
const isCfnFnImportValue = checkType('Fn::ImportValue');
const isCfnRef = checkType('Ref');
//# sourceMappingURL=data:application/json;base64,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