"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CronJob = exports.ConcurrencyPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
const workload = require("./workload");
/**
 * Concurrency policy for CronJobs.
 */
var ConcurrencyPolicy;
(function (ConcurrencyPolicy) {
    /**
     * This policy allows to run job concurrently.
     */
    ConcurrencyPolicy["ALLOW"] = "Allow";
    /**
     * This policy does not allow to run job concurrently. It does not let a new job to be scheduled if the previous one is not finished yet.
     */
    ConcurrencyPolicy["FORBID"] = "Forbid";
    /**
     * This policy replaces the currently running job if a new job is being scheduled.
     */
    ConcurrencyPolicy["REPLACE"] = "Replace";
})(ConcurrencyPolicy = exports.ConcurrencyPolicy || (exports.ConcurrencyPolicy = {}));
/**
 * A CronJob is responsible for creating a Job and scheduling it based on provided cron schedule. This helps running Jobs in a recurring manner.
 */
class CronJob extends workload.Workload {
    constructor(scope, id, props) {
        super(scope, id, {
            restartPolicy: pod_1.RestartPolicy.NEVER,
            select: false,
            ...props,
        });
        /**
         * Represents the resource type.
         */
        this.resourceType = 'cronjob';
        this.apiObject = new k8s.KubeCronJob(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        if (props.startingDeadline != undefined && props.startingDeadline.toSeconds() < 10) {
            throw new Error(`The 'startingDeadline' property cannot be less than 10 seconds since the Kubernetes CronJobController checks things every 10 seconds and hence the CronJob may not be scheduled. The value passed is: ${props.startingDeadline}`);
        }
        if (props.ttlAfterFinished != undefined && (props.successfulJobsRetained != undefined || props.failedJobsRetained != undefined)) {
            throw new Error('The \'ttlAfterFinished\' property cannot be set if \'successfulJobsRetained\' property or \'failedJobsRetained\' property is set. This would cause the retention of jobs to not work properly since it would delete the job based on its value.');
        }
        this.schedule = props.schedule;
        this.concurrencyPolicy = props.concurrencyPolicy ?? ConcurrencyPolicy.FORBID;
        this.startingDeadline = props.startingDeadline ?? cdk8s_1.Duration.seconds(10);
        this.suspend = props.suspend ?? false;
        this.successfulJobsRetained = props.successfulJobsRetained ?? 3;
        this.failedJobsRetained = props.failedJobsRetained ?? 1;
        this.jobProperties = {
            activeDeadline: props.activeDeadline,
            backoffLimit: props.backoffLimit,
            ttlAfterFinished: props.ttlAfterFinished,
        };
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            concurrencyPolicy: this.concurrencyPolicy,
            failedJobsHistoryLimit: this.failedJobsRetained,
            jobTemplate: {
                spec: this._toJobSpec(),
            },
            schedule: this.schedule.expressionString,
            startingDeadlineSeconds: this.startingDeadline.toSeconds(),
            successfulJobsHistoryLimit: this.successfulJobsRetained,
            suspend: this.suspend,
        };
    }
    /**
     * Returns the job spec.
     */
    _toJobSpec() {
        return {
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            activeDeadlineSeconds: this.jobProperties.activeDeadline?.toSeconds(),
            backoffLimit: this.jobProperties.backoffLimit,
            ttlSecondsAfterFinished: this.jobProperties.ttlAfterFinished?.toSeconds(),
        };
    }
}
exports.CronJob = CronJob;
_a = JSII_RTTI_SYMBOL_1;
CronJob[_a] = { fqn: "cdk8s-plus-22.CronJob", version: "2.0.0-rc.132" };
//# sourceMappingURL=data:application/json;base64,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