"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const ingress = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'services';
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = props.type ?? ServiceType.CLUSTER_IP;
        }
        this._externalIPs = props.externalIPs ?? [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        if (props.selector) {
            this.select(props.selector);
        }
        for (const port of props.ports ?? []) {
            this.bind(port.port, port);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        const ingr = options.ingress ?? new ingress.Ingress(this, 'Ingress');
        ingr.addRule(path, ingress.IngressBackend.fromService(this), options.pathType);
        return ingr;
    }
    /**
     * Ports for this service.
     *
     * Use `bind()` to bind additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    bind(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * Require this service to select pods matching the selector.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    select(selector) {
        const labels = selector.toPodSelectorConfig().labelSelector._toKube().matchLabels ?? {};
        for (const [key, value] of Object.entries(labels)) {
            this._selector[key] = value;
        }
    }
    /**
     * Require this service to select pods with this label.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    selectLabel(key, value) {
        this._selector[key] = value;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-22.Service", version: "2.0.0-rc.132" };
//# sourceMappingURL=data:application/json;base64,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