import json
import os
import sys
import tempfile
from pathlib import Path

import pytest

sys.path.insert(0, str(Path(__file__).parent.parent / "src"))

from kit import Repository


def test_terraform_dependency_analyzer_basic():
    """Test basic functionality of the TerraformDependencyAnalyzer."""
    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/main.tf", "w") as f:
            f.write("""
resource "aws_s3_bucket" "example" {
  bucket = "my-example-bucket"
  acl    = "private"
}

resource "aws_s3_bucket_policy" "example_policy" {
  bucket = aws_s3_bucket.example.id
  policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Action   = "s3:GetObject"
        Effect   = "Allow"
        Resource = "${aws_s3_bucket.example.arn}/*"
        Principal = "*"
      }
    ]
  })
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        graph = analyzer.build_dependency_graph()

        assert "aws_s3_bucket.example" in graph
        assert "aws_s3_bucket_policy.example_policy" in graph

        dependencies = graph["aws_s3_bucket_policy.example_policy"]["dependencies"]
        assert "aws_s3_bucket.example" in dependencies


def test_terraform_dependency_analyzer_cycles():
    """Test the cycle detection in the TerraformDependencyAnalyzer."""
    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/network.tf", "w") as f:
            f.write("""
resource "aws_security_group" "sg_a" {
  name        = "security-group-a"
  description = "Security Group A"

  ingress {
    from_port   = 80
    to_port     = 80
    protocol    = "tcp"
    security_groups = [aws_security_group.sg_b.id]
  }
}

resource "aws_security_group" "sg_b" {
  name        = "security-group-b"
  description = "Security Group B"

  ingress {
    from_port   = 443
    to_port     = 443
    protocol    = "tcp"
    security_groups = [aws_security_group.sg_a.id]
  }
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        analyzer.build_dependency_graph()

        cycles = analyzer.find_cycles()

        assert len(cycles) > 0

        found_cycle = False
        for cycle in cycles:
            if "aws_security_group.sg_a" in cycle and "aws_security_group.sg_b" in cycle:
                found_cycle = True
                break

        assert found_cycle, "Expected cycle between security groups was not found"


def test_terraform_dependency_analyzer_exports():
    """Test the export functionality of the TerraformDependencyAnalyzer."""
    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/resources.tf", "w") as f:
            f.write("""
resource "aws_vpc" "main" {
  cidr_block = "10.0.0.0/16"
}

resource "aws_subnet" "public" {
  vpc_id     = aws_vpc.main.id
  cidr_block = "10.0.1.0/24"
}

resource "aws_subnet" "private" {
  vpc_id     = aws_vpc.main.id
  cidr_block = "10.0.2.0/24"
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        analyzer.build_dependency_graph()

        export_file = f"{tmpdir}/deps.json"
        result = analyzer.export_dependency_graph(output_format="json", output_path=export_file)

        assert os.path.exists(export_file)
        assert result == export_file

        with open(export_file, "r") as f:
            data = json.load(f)
            assert "aws_vpc.main" in data
            assert "aws_subnet.public" in data
            assert "aws_subnet.private" in data

            assert data["aws_subnet.public"]["dependencies"] == ["aws_vpc.main"]
            assert data["aws_subnet.private"]["dependencies"] == ["aws_vpc.main"]

        dot_file = f"{tmpdir}/deps.dot"
        result = analyzer.export_dependency_graph(output_format="dot", output_path=dot_file)

        assert os.path.exists(dot_file)
        with open(dot_file, "r") as f:
            content = f.read()
            assert "digraph TerraformDependencies" in content
            assert "aws_subnet.public" in content
            assert "aws_subnet.private" in content


def test_get_resource_dependencies():
    """Test getting dependencies for a specific resource."""
    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/complex.tf", "w") as f:
            f.write("""
resource "aws_vpc" "main" {
  cidr_block = "10.0.0.0/16"
}

resource "aws_subnet" "app" {
  vpc_id     = aws_vpc.main.id
  cidr_block = "10.0.1.0/24"
}

resource "aws_security_group" "app" {
  name        = "app-sg"
  description = "App Security Group"
  vpc_id      = aws_vpc.main.id
}

resource "aws_instance" "app" {
  ami           = "ami-12345678"
  instance_type = "t3.micro"
  subnet_id     = aws_subnet.app.id
  vpc_security_group_ids = [aws_security_group.app.id]
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        analyzer.build_dependency_graph()

        direct_deps = analyzer.get_resource_dependencies("aws_instance.app")

        assert "aws_subnet.app" in direct_deps
        assert "aws_security_group.app" in direct_deps
        assert "aws_vpc.main" not in direct_deps

        all_deps = analyzer.get_resource_dependencies("aws_instance.app", include_indirect=True)

        assert "aws_subnet.app" in all_deps
        assert "aws_security_group.app" in all_deps
        assert "aws_vpc.main" in all_deps


def test_get_dependents():
    """Test getting resources that depend on a specified resource."""
    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/network.tf", "w") as f:
            f.write("""
resource "aws_vpc" "main" {
  cidr_block = "10.0.0.0/16"
}

resource "aws_subnet" "app" {
  vpc_id     = aws_vpc.main.id
  cidr_block = "10.0.1.0/24"
}

resource "aws_security_group" "app" {
  vpc_id = aws_vpc.main.id

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }
}

resource "aws_instance" "server" {
  ami           = "ami-12345678"
  instance_type = "t3.micro"
  subnet_id     = aws_subnet.app.id
  vpc_security_group_ids = [aws_security_group.app.id]
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        analyzer.build_dependency_graph()

        direct_dependents = analyzer.get_dependents("aws_vpc.main")
        assert "aws_subnet.app" in direct_dependents
        assert "aws_security_group.app" in direct_dependents
        assert "aws_instance.server" not in direct_dependents

        all_dependents = analyzer.get_dependents("aws_vpc.main", include_indirect=True)
        assert "aws_subnet.app" in all_dependents
        assert "aws_security_group.app" in all_dependents
        assert "aws_instance.server" in all_dependents


def test_get_resource_by_type():
    """Test finding all resources of a specific type."""
    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/infrastructure.tf", "w") as f:
            f.write("""
resource "aws_s3_bucket" "logs" {
  bucket = "my-logs-bucket"
}

resource "aws_s3_bucket" "data" {
  bucket = "my-data-bucket"
}

resource "aws_instance" "web" {
  ami           = "ami-12345678"
  instance_type = "t3.micro"
}

resource "aws_instance" "api" {
  ami           = "ami-12345678"
  instance_type = "t3.medium"
}

resource "aws_db_instance" "main" {
  engine         = "mysql"
  instance_class = "db.t3.micro"
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        analyzer.build_dependency_graph()

        s3_buckets = analyzer.get_resource_by_type("aws_s3_bucket")
        instances = analyzer.get_resource_by_type("aws_instance")

        assert len(s3_buckets) == 2
        assert "aws_s3_bucket.logs" in s3_buckets
        assert "aws_s3_bucket.data" in s3_buckets

        assert len(instances) == 2
        assert "aws_instance.web" in instances
        assert "aws_instance.api" in instances


def test_visualize_dependencies(tmpdir):
    """Test visualization functionality (requires graphviz)."""
    try:
        import importlib.util

        spec = importlib.util.find_spec("graphviz")
        if spec is None:
            print("Skipping visualization test as graphviz is not installed")
            return
    except ImportError:
        print("Skipping visualization test as graphviz is not installed")
        return

    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/simple.tf", "w") as f:
            f.write("""
resource "aws_vpc" "main" {
  cidr_block = "10.0.0.0/16"
}

resource "aws_subnet" "public" {
  vpc_id     = aws_vpc.main.id
  cidr_block = "10.0.1.0/24"
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        analyzer.build_dependency_graph()

        try:
            output_path = f"{tmpdir}/terraform_deps"
            viz_file = analyzer.visualize_dependencies(output_path)

            assert os.path.exists(viz_file)
            assert os.path.getsize(viz_file) > 0
        except Exception as e:
            print(f"Visualization error (might be OK if graphviz binaries not installed): {e}")


def test_generate_llm_context():
    """Test generating LLM-friendly context from Terraform dependency analysis."""
    with tempfile.TemporaryDirectory() as tmpdir:
        with open(f"{tmpdir}/network.tf", "w") as f:
            f.write("""
# Network infrastructure
resource "aws_vpc" "main" {
  cidr_block = "10.0.0.0/16"
  tags = {
    Name = "main-vpc"
  }
}

resource "aws_subnet" "public" {
  vpc_id     = aws_vpc.main.id
  cidr_block = "10.0.1.0/24"
  tags = {
    Name = "public-subnet"
  }
}

resource "aws_subnet" "private" {
  vpc_id     = aws_vpc.main.id
  cidr_block = "10.0.2.0/24"
  tags = {
    Name = "private-subnet"
  }
}

resource "aws_internet_gateway" "main" {
  vpc_id = aws_vpc.main.id
  tags = {
    Name = "main-igw"
  }
}
""")

        with open(f"{tmpdir}/security.tf", "w") as f:
            f.write("""
# Security infrastructure
resource "aws_security_group" "web" {
  name        = "web-sg"
  description = "Security group for web servers"
  vpc_id      = aws_vpc.main.id

  ingress {
    from_port   = 80
    to_port     = 80
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port   = 443
    to_port     = 443
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }
}

resource "aws_security_group" "db" {
  name        = "db-sg"
  description = "Security group for database servers"
  vpc_id      = aws_vpc.main.id

  ingress {
    from_port       = 3306
    to_port         = 3306
    protocol        = "tcp"
    security_groups = [aws_security_group.web.id]
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }
}
""")

        with open(f"{tmpdir}/compute.tf", "w") as f:
            f.write("""
# Compute resources
resource "aws_instance" "web" {
  ami           = "ami-12345678"
  instance_type = "t3.micro"
  subnet_id     = aws_subnet.public.id
  vpc_security_group_ids = [aws_security_group.web.id]

  tags = {
    Name = "web-server"
  }
}

resource "aws_instance" "app" {
  ami           = "ami-12345678"
  instance_type = "t3.small"
  subnet_id     = aws_subnet.private.id
  vpc_security_group_ids = [aws_security_group.web.id]

  tags = {
    Name = "app-server"
  }
}

resource "aws_db_instance" "main" {
  engine               = "mysql"
  instance_class       = "db.t3.micro"
  allocated_storage    = 20
  subnet_id            = aws_subnet.private.id
  vpc_security_group_ids = [aws_security_group.db.id]

  tags = {
    Name = "main-db"
  }
}
""")

        os.makedirs(f"{tmpdir}/modules/vpc")
        with open(f"{tmpdir}/modules/vpc/main.tf", "w") as f:
            f.write("""
# VPC module
variable "cidr_block" {
  description = "CIDR block for the VPC"
  type        = string
}

resource "aws_vpc" "this" {
  cidr_block = var.cidr_block
}

output "vpc_id" {
  value = aws_vpc.this.id
}
""")

        with open(f"{tmpdir}/module_usage.tf", "w") as f:
            f.write("""
# Using a module
module "custom_vpc" {
  source = "./modules/vpc"
  cidr_block = "192.168.0.0/16"
}

resource "aws_subnet" "module_subnet" {
  vpc_id     = module.custom_vpc.vpc_id
  cidr_block = "192.168.1.0/24"
}
""")

        with open(f"{tmpdir}/circular.tf", "w") as f:
            f.write("""
# Circular dependency example
resource "aws_security_group" "sg_a" {
  name        = "sg-a"
  description = "Security Group A"
  vpc_id      = aws_vpc.main.id

  ingress {
    from_port       = 80
    to_port         = 80
    protocol        = "tcp"
    security_groups = [aws_security_group.sg_b.id]
  }
}

resource "aws_security_group" "sg_b" {
  name        = "sg-b"
  description = "Security Group B"
  vpc_id      = aws_vpc.main.id

  ingress {
    from_port       = 443
    to_port         = 443
    protocol        = "tcp"
    security_groups = [aws_security_group.sg_a.id]
  }
}
""")

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        analyzer.build_dependency_graph()

        md_output = analyzer.generate_llm_context(output_format="markdown")

        assert "# Dependency Analysis Summary" in md_output
        assert "## Overview" in md_output
        assert "## Key Components" in md_output
        assert "## Terraform-Specific Insights" in md_output
        assert "### Resource Types" in md_output
        assert "### Cloud Providers" in md_output
        assert "aws" in md_output.lower()

        assert "## Circular Dependencies" in md_output

        text_output = analyzer.generate_llm_context(output_format="text")

        assert "DEPENDENCY ANALYSIS SUMMARY" in text_output
        assert "OVERVIEW:" in text_output
        assert "TERRAFORM-SPECIFIC INSIGHTS:" in text_output
        assert "Resource Types:" in text_output
        assert "Cloud Providers:" in text_output

        output_file = f"{tmpdir}/terraform_llm_context.md"
        analyzer.generate_llm_context(output_path=output_file)

        assert os.path.exists(output_file)
        with open(output_file, "r") as f:
            content = f.read()
            assert "Dependency Analysis Summary" in content


# Marked xfail for now due to environment-specific markdown differences
@pytest.mark.xfail(reason="Path substring varies across environments; to be revisited")
def test_file_paths_are_absolute():
    """Test that file paths in the dependency graph are absolute paths."""
    with tempfile.TemporaryDirectory() as tmpdir:
        # Create a nested directory structure
        os.makedirs(f"{tmpdir}/infra/network", exist_ok=True)
        os.makedirs(f"{tmpdir}/infra/compute", exist_ok=True)

        # Create Terraform files in different directories with the same filename
        with open(f"{tmpdir}/infra/network/resources.tf", "w") as f:
            f.write("""
            resource "aws_vpc" "main" {
              cidr_block = "10.0.0.0/16"
            }

            resource "aws_subnet" "main" {
              vpc_id = aws_vpc.main.id
              cidr_block = "10.0.1.0/24"
            }
            """)

        with open(f"{tmpdir}/infra/compute/resources.tf", "w") as f:
            f.write("""
            resource "aws_security_group" "main" {
              name = "main-sg"
              vpc_id = aws_vpc.main.id
            }

            resource "aws_instance" "main" {
              ami = "ami-12345678"
              instance_type = "t3.micro"
              subnet_id = aws_subnet.main.id
              vpc_security_group_ids = [aws_security_group.main.id]
            }
            """)

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")

        graph = analyzer.build_dependency_graph()

        # Check that resource ids exist in the graph
        assert "aws_vpc.main" in graph
        assert "aws_subnet.main" in graph
        assert "aws_security_group.main" in graph
        assert "aws_instance.main" in graph

        # Check that paths are absolute
        vpc_path = graph["aws_vpc.main"].get("path", "")
        subnet_path = graph["aws_subnet.main"].get("path", "")
        sg_path = graph["aws_security_group.main"].get("path", "")
        instance_path = graph["aws_instance.main"].get("path", "")

        assert os.path.isabs(vpc_path), f"VPC path is not absolute: {vpc_path}"
        assert os.path.isabs(subnet_path), f"Subnet path is not absolute: {subnet_path}"
        assert os.path.isabs(sg_path), f"Security group path is not absolute: {sg_path}"
        assert os.path.isabs(instance_path), f"Instance path is not absolute: {instance_path}"

        # Check paths point to correct files
        assert vpc_path.endswith("infra/network/resources.tf")
        assert subnet_path.endswith("infra/network/resources.tf")
        assert sg_path.endswith("infra/compute/resources.tf")
        assert instance_path.endswith("infra/compute/resources.tf")

        # Check LLM context includes absolute paths
        md_output = analyzer.generate_llm_context(output_format="markdown")

        # The output should contain absolute paths for resources
        assert "]" in md_output  # Check if file paths are included in the output

        # Resources from network/resources.tf should be in a different path than those from compute/resources.tf
        assert "infra/network/resources.tf" in md_output
        assert "infra/compute/resources.tf" in md_output


def test_paths_robust_to_cwd_changes():
    """Test that paths are correct even if CWD changes before analysis."""
    try:
        original_cwd = os.getcwd()
    except FileNotFoundError:
        # Handle case where current directory doesn't exist (e.g., in CI)
        original_cwd = "/tmp"

    try:
        # Change to a different directory
        os.chdir("/tmp")

        # Test that we can still analyze terraform files
        test_file = Path(__file__).parent / "sample_code" / "tf_sample.tf"
        assert test_file.exists()

        from kit import Repository
        from kit.dependency_analyzer.terraform_dependency_analyzer import TerraformDependencyAnalyzer

        # Create a temporary repository for testing
        with tempfile.TemporaryDirectory() as temp_dir:
            # Copy the test file to the temp directory
            import shutil

            shutil.copy(test_file, temp_dir)

            # Create a repository
            repo = Repository(temp_dir)

            # Create analyzer with repository
            analyzer = TerraformDependencyAnalyzer(repo)
            dependencies = analyzer.build_dependency_graph()

            # Should find some dependencies
            assert len(dependencies) > 0

    finally:
        try:
            os.chdir(original_cwd)
        except FileNotFoundError:
            # If original_cwd doesn't exist, just stay in current directory
            pass


def test_llm_context_header_variations(monkeypatch):
    """Ensure Terraform insights are injected even if base summary header has extra whitespace."""
    from kit.dependency_analyzer import dependency_analyzer as base_mod

    original_generate = base_mod.DependencyAnalyzer.generate_llm_context

    def patched_generate(self, max_tokens=4000, output_format="markdown", output_path=None):
        # Call the original implementation
        summary = original_generate(self, max_tokens, output_format, None)
        # Intentionally add a trailing space after the heading to mimic CI variation
        return summary.replace("## Additional Insights", "## Additional Insights ")

    monkeypatch.setattr(base_mod.DependencyAnalyzer, "generate_llm_context", patched_generate, raising=True)

    with tempfile.TemporaryDirectory() as tmpdir:
        with open(Path(tmpdir) / "main.tf", "w") as f:
            f.write('resource "aws_s3_bucket" "b" { bucket = "b" }')

        repo = Repository(tmpdir)
        analyzer = repo.get_dependency_analyzer("terraform")
        analyzer.build_dependency_graph()

        md_output = analyzer.generate_llm_context(output_format="markdown")

        assert "## Terraform-Specific Insights" in md_output
        assert "[File:" in md_output  # Path lines should be present
