"""
update column types

Revision ID: 51adc521f5d4
Revises: c76cf026488e
Create Date: 2025-09-18 21:25:08.510969

"""

from collections.abc import Sequence

import sqlalchemy as sa
from sqlalchemy.dialects import postgresql

from alembic import op

# revision identifiers, used by Alembic.
revision: str = "51adc521f5d4"
down_revision: str | Sequence[str] | None = "c76cf026488e"
branch_labels: str | Sequence[str] | None = None
depends_on: str | Sequence[str] | None = None


def upgrade() -> None:
    """Upgrade schema."""
    bind = op.get_bind()
    bind.execute(sa.text("CREATE EXTENSION IF NOT EXISTS timescaledb;"))

    # Disable compression on hypertables before altering column types
    tables_with_compression = [
        "change_log_committed",
        "time_product_features",
        "time_product_features_text",
        "time_region_features",
        "time_region_features_text",
        "time_sku_features",
        "time_sku_features_text",
        "time_store_features_text",
    ]

    for table in tables_with_compression:
        bind.execute(sa.text(f"ALTER TABLE {table} SET (timescaledb.compress = false);"))

    # ### commands auto generated by Alembic - please adjust! ###
    op.create_table(
        "time_store_features",
        sa.Column("company_id", sa.Integer(), nullable=False),
        sa.Column("store_id", sa.Integer(), nullable=False),
        sa.Column("feature_id", sa.Integer(), nullable=False),
        sa.Column("ts", sa.Date(), nullable=False),
        sa.Column("value", sa.DOUBLE_PRECISION(), nullable=True),
        sa.ForeignKeyConstraint(
            ["company_id", "feature_id"],
            ["feature_descriptions.company_id", "feature_descriptions.feature_id"],
            name="fk_tsf_features",
            onupdate="RESTRICT",
            ondelete="CASCADE",
        ),
        sa.ForeignKeyConstraint(
            ["company_id", "store_id"],
            ["stores.company_id", "stores.store_id"],
            name="fk_tsf_stores",
            onupdate="RESTRICT",
            ondelete="CASCADE",
        ),
        sa.PrimaryKeyConstraint("company_id", "store_id", "feature_id", "ts", name="pk_time_store_features"),
        info={
            "timescale": {
                "time_column": "ts",
                "space_column": "store_id",
                "number_partitions": 32,
                "chunk_interval": "14 days",
                "compression": {
                    "enabled": True,
                    "segmentby": ["company_id", "store_id", "feature_id"],
                    "orderby": "ts DESC",
                    "compress_after": "30 days",
                },
                "caggs": [],
            }
        },
    )
    op.create_index(
        "ix_tsf_company_feature_ts", "time_store_features", ["company_id", "feature_id", "ts"], unique=False
    )
    op.create_index("ix_tsf_company_store_ts", "time_store_features", ["company_id", "store_id", "ts"], unique=False)
    op.alter_column(
        "change_log_committed",
        "valid_from",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )
    op.alter_column(
        "time_product_features",
        "ts",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )
    op.alter_column(
        "time_product_features_text",
        "ts",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )
    op.alter_column(
        "time_region_features",
        "ts",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )
    op.alter_column(
        "time_region_features_text",
        "ts",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )
    op.alter_column(
        "time_sku_features",
        "ts",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )
    op.alter_column(
        "time_sku_features_text",
        "ts",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )
    op.alter_column(
        "time_store_features_text",
        "ts",
        existing_type=postgresql.TIMESTAMP(timezone=True),
        type_=sa.Date(),
        existing_nullable=False,
    )

    # TimescaleDB setup for new time_store_features table
    bind.execute(
        sa.text("""
      SELECT create_hypertable(
        'time_store_features', 'ts',
        partitioning_column => 'store_id',
        number_partitions   => 32,
        create_default_indexes => FALSE,
        if_not_exists => TRUE
      );
    """)
    )
    bind.execute(sa.text("SELECT set_chunk_time_interval('time_store_features', INTERVAL '14 days');"))
    bind.execute(
        sa.text("""
      ALTER TABLE "time_store_features"
      SET (timescaledb.compress = 'on',
           timescaledb.compress_orderby  = 'ts DESC',
           timescaledb.compress_segmentby = 'company_id,store_id,feature_id');
    """)
    )
    bind.execute(sa.text("SELECT add_compression_policy('time_store_features', INTERVAL '30 days');"))

    # Re-enable compression on existing hypertables after column type changes
    compression_configs = {
        "change_log_committed": "company_id,sku_id,delivery_day",
        "time_product_features": "company_id,product_id,feature_id",
        "time_product_features_text": "company_id,product_id,feature_id",
        "time_region_features": "company_id,region_id,feature_id",
        "time_region_features_text": "company_id,region_id,feature_id",
        "time_sku_features": "company_id,sku_id,feature_id",
        "time_sku_features_text": "company_id,sku_id,feature_id",
        "time_store_features_text": "company_id,store_id,feature_id",
    }

    for table, segmentby in compression_configs.items():
        time_col = (
            "ts" if table.startswith("time_") else "valid_from" if table == "change_log_committed" else "order_day"
        )
        bind.execute(
            sa.text(f"""
          ALTER TABLE "{table}"
          SET (timescaledb.compress = 'on',
               timescaledb.compress_orderby = '{time_col} DESC',
               timescaledb.compress_segmentby = '{segmentby}');
        """)
        )

    # ### end Alembic commands ###


def downgrade() -> None:
    """Downgrade schema."""
    # ### commands auto generated by Alembic - please adjust! ###
    op.alter_column(
        "time_store_features_text",
        "ts",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.alter_column(
        "time_sku_features_text",
        "ts",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.alter_column(
        "time_sku_features",
        "ts",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.alter_column(
        "time_region_features_text",
        "ts",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.alter_column(
        "time_region_features",
        "ts",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.alter_column(
        "time_product_features_text",
        "ts",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.alter_column(
        "time_product_features",
        "ts",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.alter_column(
        "change_log_committed",
        "valid_from",
        existing_type=sa.Date(),
        type_=postgresql.TIMESTAMP(timezone=True),
        existing_nullable=False,
    )
    op.drop_index("ix_tsf_company_store_ts", table_name="time_store_features")
    op.drop_index("ix_tsf_company_feature_ts", table_name="time_store_features")
    op.drop_table("time_store_features")
    # ### end Alembic commands ###
