//! `SeaORM` Entity, @generated by sea-orm-codegen 1.1.0

use sea_orm::entity::prelude::*;
use sea_orm::{DatabaseTransaction, DbErr, ActiveValue};
use tracing::info;
use super::solid_queue_ready_executions;


// Define the Retryable trait
pub trait Retryable {
    /// Retry a failed job by moving it from failed_executions to ready_executions
    async fn retry(&self, txn: &DatabaseTransaction) -> Result<(), DbErr>;

    /// Retry all failed jobs by moving them from failed_executions to ready_executions
    async fn retry_all(&self, txn: &DatabaseTransaction) -> Result<u64, DbErr>;
}

// Define the Discardable trait
pub trait Discardable {
    /// Discard a failed job by removing it from failed_executions and optionally marking the job as finished
    async fn discard(&self, txn: &DatabaseTransaction) -> Result<(), DbErr>;

    /// Discard all failed jobs by removing them from failed_executions and optionally marking the jobs as finished
    async fn discard_all(&self, txn: &DatabaseTransaction) -> Result<u64, DbErr>;
}

#[derive(Clone, Debug, PartialEq, DeriveEntityModel, Eq)]
#[sea_orm(table_name = "solid_queue_failed_executions")]
pub struct Model {
    #[sea_orm(primary_key)]
    pub id: i64,
    #[sea_orm(unique)]
    pub job_id: i64,
    #[sea_orm(column_type = "Text", nullable)]
    pub error: Option<String>,
    pub created_at: DateTime,
}

#[derive(Copy, Clone, Debug, EnumIter, DeriveRelation)]
pub enum Relation {
    #[sea_orm(
        belongs_to = "super::solid_queue_jobs::Entity",
        from = "Column::JobId",
        to = "super::solid_queue_jobs::Column::Id",
        on_update = "NoAction",
        on_delete = "Cascade"
    )]
    SolidQueueJobs,
}

impl Related<super::solid_queue_jobs::Entity> for Entity {
    fn to() -> RelationDef {
        Relation::SolidQueueJobs.def()
    }
}

impl ActiveModelBehavior for ActiveModel {}

// Implement Retryable for Model
impl Retryable for Model {
    async fn retry(&self, txn: &DatabaseTransaction) -> Result<(), DbErr> {
        // 1. Find failed job record
        let job_result = super::solid_queue_jobs::Entity::find_by_id(self.job_id)
            .one(txn)
            .await?;

        if let Some(job) = job_result {
            // 2. Add job to ready_executions table
            let ready_execution = solid_queue_ready_executions::ActiveModel {
                id: ActiveValue::NotSet,
                queue_name: ActiveValue::Set(job.queue_name.clone()),
                job_id: ActiveValue::Set(job.id),
                priority: ActiveValue::Set(job.priority),
                created_at: ActiveValue::Set(chrono::Utc::now().naive_utc()),
            };
            ready_execution.insert(txn).await?;

            // 3. Delete failed execution record
            Entity::delete_many()
                .filter(Column::JobId.eq(self.job_id))
                .exec(txn)
                .await?;

            info!("Retried failed job {}", self.job_id);
            Ok(())
        } else {
            Err(DbErr::Custom(format!("Job with ID {} not found", self.job_id)))
        }
    }

    async fn retry_all(&self, txn: &DatabaseTransaction) -> Result<u64, DbErr> {
        // 1. Get all failed job records
        let failed_executions = Entity::find()
            .all(txn)
            .await?;

        if failed_executions.is_empty() {
            return Ok(0);
        }

        let job_ids: Vec<i64> = failed_executions.iter()
            .map(|execution| execution.job_id)
            .collect();

        // 2. Get all related job information
        let jobs = super::solid_queue_jobs::Entity::find()
            .filter(super::solid_queue_jobs::Column::Id.is_in(job_ids.clone()))
            .all(txn)
            .await?;

        // 3. Add all jobs to ready_executions table
        for job in &jobs {
            let ready_execution = solid_queue_ready_executions::ActiveModel {
                id: ActiveValue::NotSet,
                queue_name: ActiveValue::Set(job.queue_name.clone()),
                job_id: ActiveValue::Set(job.id),
                priority: ActiveValue::Set(job.priority),
                created_at: ActiveValue::Set(chrono::Utc::now().naive_utc()),
            };
            ready_execution.insert(txn).await?;
        }

        // 4. Delete all failed execution records
        let delete_result = Entity::delete_many()
            .filter(Column::JobId.is_in(job_ids))
            .exec(txn)
            .await?;

        let count = delete_result.rows_affected;
        info!("Retried all {} failed jobs", jobs.len());
        Ok(count)
    }
}

// Implement Discardable for Model
impl Discardable for Model {
    async fn discard(&self, txn: &DatabaseTransaction) -> Result<(), DbErr> {
        // 1. Find job record
        let job_result = super::solid_queue_jobs::Entity::find_by_id(self.job_id)
            .one(txn)
            .await?;

        if let Some(job) = job_result {
            // 2. Update job status to completed
            let mut job_model: super::solid_queue_jobs::ActiveModel = job.into();
            job_model.finished_at = ActiveValue::Set(Some(chrono::Utc::now().naive_utc()));
            job_model.updated_at = ActiveValue::Set(chrono::Utc::now().naive_utc());
            job_model.update(txn).await?;

            // 3. Delete failed execution record
            Entity::delete_many()
                .filter(Column::JobId.eq(self.job_id))
                .exec(txn)
                .await?;

            info!("Discarded failed job {}", self.job_id);
            Ok(())
        } else {
            Err(DbErr::Custom(format!("Job with ID {} not found", self.job_id)))
        }
    }

    async fn discard_all(&self, txn: &DatabaseTransaction) -> Result<u64, DbErr> {
        // 1. Get all failed job records
        let failed_executions = Entity::find()
            .all(txn)
            .await?;

        if failed_executions.is_empty() {
            return Ok(0);
        }

        let job_ids: Vec<i64> = failed_executions.iter()
            .map(|execution| execution.job_id)
            .collect();

        let now = chrono::Utc::now().naive_utc();

        // 2. Update all related jobs to completed status
        for job_id in &job_ids {
            let job_result = super::solid_queue_jobs::Entity::find_by_id(*job_id)
                .one(txn)
                .await?;

            if let Some(job) = job_result {
                let mut job_model: super::solid_queue_jobs::ActiveModel = job.into();
                job_model.finished_at = ActiveValue::Set(Some(now));
                job_model.updated_at = ActiveValue::Set(now);
                job_model.update(txn).await?;
            }
        }

        // 3. Delete all failed execution records
        let delete_result = Entity::delete_many()
            .filter(Column::JobId.is_in(job_ids))
            .exec(txn)
            .await?;

        let count = delete_result.rows_affected;
        info!("Discarded all {} failed jobs", count);
        Ok(count)
    }
}

// Implement Retryable for Entity
impl Retryable for Entity {
    async fn retry(&self, _txn: &DatabaseTransaction) -> Result<(), DbErr> {
        // Since Entity doesn't have a specific job_id, this method needs a different implementation
        Err(DbErr::Custom("Cannot retry a job from Entity, please use a Model instance instead".to_string()))
    }

    async fn retry_all(&self, txn: &DatabaseTransaction) -> Result<u64, DbErr> {
        // 1. Get all failed job records
        let failed_executions = Entity::find()
            .all(txn)
            .await?;

        if failed_executions.is_empty() {
            return Ok(0);
        }

        let job_ids: Vec<i64> = failed_executions.iter()
            .map(|execution| execution.job_id)
            .collect();

        // 2. Get all related job information
        let jobs = super::solid_queue_jobs::Entity::find()
            .filter(super::solid_queue_jobs::Column::Id.is_in(job_ids.clone()))
            .all(txn)
            .await?;

        // 3. Add all jobs to ready_executions table
        for job in &jobs {
            let ready_execution = solid_queue_ready_executions::ActiveModel {
                id: ActiveValue::NotSet,
                queue_name: ActiveValue::Set(job.queue_name.clone()),
                job_id: ActiveValue::Set(job.id),
                priority: ActiveValue::Set(job.priority),
                created_at: ActiveValue::Set(chrono::Utc::now().naive_utc()),
            };
            ready_execution.insert(txn).await?;
        }

        // 4. Delete all failed execution records
        let delete_result = Entity::delete_many()
            .filter(Column::JobId.is_in(job_ids))
            .exec(txn)
            .await?;

        let count = delete_result.rows_affected;
        info!("Retried all {} failed jobs", jobs.len());
        Ok(count)
    }
}

// Implement Discardable for Entity
impl Discardable for Entity {
    async fn discard(&self, _txn: &DatabaseTransaction) -> Result<(), DbErr> {
        // Since Entity doesn't have a specific job_id, this method needs a different implementation
        Err(DbErr::Custom("Cannot discard a job from Entity, please use a Model instance instead".to_string()))
    }

    async fn discard_all(&self, txn: &DatabaseTransaction) -> Result<u64, DbErr> {
        // 1. Get all failed job records
        let failed_executions = Entity::find()
            .all(txn)
            .await?;

        if failed_executions.is_empty() {
            return Ok(0);
        }

        let job_ids: Vec<i64> = failed_executions.iter()
            .map(|execution| execution.job_id)
            .collect();

        let now = chrono::Utc::now().naive_utc();

        // 2. Update all related jobs to completed status
        for job_id in &job_ids {
            let job_result = super::solid_queue_jobs::Entity::find_by_id(*job_id)
                .one(txn)
                .await?;

            if let Some(job) = job_result {
                let mut job_model: super::solid_queue_jobs::ActiveModel = job.into();
                job_model.finished_at = ActiveValue::Set(Some(now));
                job_model.updated_at = ActiveValue::Set(now);
                job_model.update(txn).await?;
            }
        }

        // 3. Delete all failed execution records
        let delete_result = Entity::delete_many()
            .filter(Column::JobId.is_in(job_ids))
            .exec(txn)
            .await?;

        let count = delete_result.rows_affected;
        info!("Discarded all {} failed jobs", count);
        Ok(count)
    }
}
