import datetime
import json
from dataclasses import dataclass
from pathlib import Path


@dataclass
class Mark:
    """Оценка за отдельное задание.

    points: int - количество набранных баллов
    max_points: int - максимальное количество баллов за задание
    note: str - комментарий (опционально)
    iso_timestamp: str - метка времени в формате ISO (создается автоматически)
    """

    points: int  # сколько баллов получил
    max_points: int  # какое было максимальное количество баллов
    note: str = ""
    iso_timestamp: str = ""

    @property
    def score(self):
        return round(self.points / self.max_points * 100)

    def __post_init__(self):
        if not self.iso_timestamp:
            self.iso_timestamp = datetime.datetime.now().isoformat()  # добавить зону

    def save(self, path):
        content = json.dumps(self.__dict__, ensure_ascii=False)
        Path(path).write_text(content, encoding="utf-8")

    @classmethod
    def load(cls, path):
        return Mark(**json.loads(Path(path).read_text(encoding="utf-8")))


@dataclass
class User:
    """Пользователь с уникальным идентификатором и набором выполненных заданий."""

    uuid: str
    marks: list[Mark]

    @classmethod
    def new(cls):
        from uuid import uuid1

        return User(str(uuid1())[0:6], list())

    def register(self, path):
        """Прочитать файл с оценкой за задание."""
        mark = Mark.load(path)
        self.marks.append(mark)

    @property
    def summary(self) -> Mark:
        points = 0
        max_points = 0
        for m in self.marks:
            points += m.points
            max_points += m.max_points
        return Mark(points, max_points, "суммарная оценка")

    @property
    def notes(self):
        return [m.note for m in self.marks]
