"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambda'));
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL',
            },
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.defaultChild.cfnOptions.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.ref,
                beforeAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.stringValue({ produce: () => this.postHook && this.postHook.functionName }),
            },
        };
    }
    /**
     * Import an Lambda Deployment Group defined either outside the CDK app, or in a different AWS region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * Associate a function to run before deployment begins.
     * @param preHook function to run before deployment beings
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * Associate a function to run after deployment completes.
     * @param postHook function to run after deployment completes
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus
     * on this deployment group resource.
     * @param grantee to grant permission to
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
class ImportedLambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
    }
}
//# sourceMappingURL=data:application/json;base64,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